<?php
/**
 * Core conversion logic for VLWP WebP.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_WebP_Core {
	/**
	 * Singleton instance.
	 *
	 * @var VLWP_WebP_Core|null
	 */
	private static $instance = null;

	/**
	 * Option key for disable flag.
	 *
	 * @var string
	 */
	const OPTION_DISABLE_CONVERSION = 'vlwp_webp_disable_conversion';

	/**
	 * Option key for allowed image types.
	 *
	 * @var string
	 */
	const OPTION_ALLOWED_TYPES = 'vlwp_webp_allowed_types';

	/**
	 * Option key for background conversion of previously uploaded images.
	 *
	 * @var string
	 */
	const OPTION_CONVERT_EXISTING = 'vlwp_webp_convert_existing';

	/**
	 * Option key for legacy scan cursor.
	 *
	 * @var string
	 */
	const OPTION_LEGACY_SCAN_CURSOR = 'vlwp_webp_legacy_scan_cursor';

	/**
	 * Option key for per-site network override toggle.
	 *
	 * @var string
	 */
	const OPTION_USE_NETWORK_DEFAULTS = 'vlwp_webp_use_network_defaults';

	/**
	 * Site option key for network disable default.
	 *
	 * @var string
	 */
	const NETWORK_OPTION_DISABLE_CONVERSION = 'vlwp_webp_network_disable_conversion';

	/**
	 * Site option key for network allowed types default.
	 *
	 * @var string
	 */
	const NETWORK_OPTION_ALLOWED_TYPES = 'vlwp_webp_network_allowed_types';

	/**
	 * Option key for queue payload.
	 *
	 * @var string
	 */
	const OPTION_QUEUE = 'vlwp_webp_queue';

	/**
	 * Option key for stats payload.
	 *
	 * @var string
	 */
	const OPTION_STATS = 'vlwp_webp_stats';

	/**
	 * Cron hook for queue processing.
	 *
	 * @var string
	 */
	const CRON_HOOK = 'vlwp_webp_process_queue';

	/**
	 * Cron hook for scanning old uploads.
	 *
	 * @var string
	 */
	const LEGACY_SCAN_CRON_HOOK = 'vlwp_webp_scan_legacy_uploads';

	/**
	 * Queue threshold in bytes (5MB).
	 *
	 * @var int
	 */
	const ASYNC_THRESHOLD_BYTES = 5242880;

	/**
	 * Max queue items per cron run.
	 *
	 * @var int
	 */
	const MAX_QUEUE_BATCH = 4;

	/**
	 * Max legacy attachments scanned per run.
	 *
	 * @var int
	 */
	const LEGACY_SCAN_BATCH = 5;

	/**
	 * Delay between legacy scan runs.
	 *
	 * @var int
	 */
	const LEGACY_SCAN_INTERVAL_SECONDS = 600;

	/**
	 * Default allowed types.
	 *
	 * @var string[]
	 */
	const DEFAULT_ALLOWED_TYPES = array( 'jpg', 'jpeg', 'png' );

	/**
	 * Conversion quality.
	 *
	 * @var int
	 */
	const WEBP_QUALITY = 82;

	/**
	 * Get singleton.
	 *
	 * @return VLWP_WebP_Core
	 */
	public static function vlwp_get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Init hooks.
	 */
	public function vlwp_init() {
		add_filter( 'wp_generate_attachment_metadata', array( $this, 'vlwp_convert_attachment_metadata' ), 10, 2 );
		add_filter( 'wp_get_attachment_image_src', array( $this, 'vlwp_prefer_webp_src' ), 10, 3 );
		add_action( self::CRON_HOOK, array( $this, 'vlwp_process_queue' ) );
		add_action( self::LEGACY_SCAN_CRON_HOOK, array( $this, 'vlwp_scan_legacy_uploads' ) );

		$this->vlwp_maybe_schedule_legacy_scan();
	}

	/**
	 * Convert attachment and generated sizes to WebP when enabled.
	 *
	 * @param array $metadata Attachment metadata.
	 * @param int   $attachment_id Attachment ID.
	 * @return array
	 */
	public function vlwp_convert_attachment_metadata( $metadata, $attachment_id ) {
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			return $metadata;
		}

		if ( ! $this->vlwp_is_conversion_enabled() ) {
			return $metadata;
		}

		if ( empty( $metadata['file'] ) ) {
			return $metadata;
		}

		$original_file = get_attached_file( $attachment_id );
		if ( empty( $original_file ) || ! file_exists( $original_file ) ) {
			return $metadata;
		}

		if ( ! $this->vlwp_is_extension_allowed( $original_file ) ) {
			return $metadata;
		}

		if ( filesize( $original_file ) > self::ASYNC_THRESHOLD_BYTES ) {
			$this->vlwp_enqueue_attachment( $attachment_id );
			return $metadata;
		}

		return $this->vlwp_convert_attachment_now( $metadata, $attachment_id, $original_file );
	}

	/**
	 * Process queue entries in background.
	 */
	public function vlwp_process_queue() {
		$queue = get_option( self::OPTION_QUEUE, array() );
		if ( ! is_array( $queue ) || empty( $queue ) ) {
			return;
		}

		$batch = array_splice( $queue, 0, self::MAX_QUEUE_BATCH );
		update_option( self::OPTION_QUEUE, array_values( array_unique( $queue ) ), false );

		foreach ( $batch as $attachment_id ) {
			$attachment_id = (int) $attachment_id;

			if ( $attachment_id <= 0 || ! wp_attachment_is_image( $attachment_id ) ) {
				$this->vlwp_increment_stats( array( 'queue_failed' => 1 ) );
				continue;
			}

			if ( ! $this->vlwp_is_conversion_enabled() ) {
				$this->vlwp_increment_stats( array( 'queue_failed' => 1 ) );
				continue;
			}

			$metadata = wp_get_attachment_metadata( $attachment_id );
			$original_file = get_attached_file( $attachment_id );

			if ( empty( $original_file ) || ! file_exists( $original_file ) || ! $this->vlwp_is_extension_allowed( $original_file ) ) {
				$this->vlwp_increment_stats( array( 'queue_failed' => 1 ) );
				continue;
			}

			if ( ! is_array( $metadata ) || empty( $metadata['file'] ) ) {
				$metadata = array(
					'file'  => _wp_relative_upload_path( $original_file ),
					'sizes' => array(),
				);
			}

			$updated_metadata = $this->vlwp_convert_attachment_now( $metadata, $attachment_id, $original_file );
			wp_update_attachment_metadata( $attachment_id, $updated_metadata );

			$this->vlwp_increment_stats(
				array(
					'queue_processed' => 1,
				)
			);
		}

		if ( ! empty( $queue ) ) {
			$this->vlwp_schedule_queue_processing();
		}
	}

	/**
	 * Scan old uploads and enqueue eligible images in low volume.
	 */
	public function vlwp_scan_legacy_uploads() {
		if ( ! $this->vlwp_should_scan_legacy_uploads() ) {
			return;
		}

		$cursor = (int) get_option( self::OPTION_LEGACY_SCAN_CURSOR, 0 );
		global $wpdb;

		$ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT ID FROM {$wpdb->posts}
				WHERE post_type = %s
					AND post_status = %s
					AND post_mime_type LIKE %s
					AND ID > %d
				ORDER BY ID ASC
				LIMIT %d",
				'attachment',
				'inherit',
				'image/%',
				$cursor,
				self::LEGACY_SCAN_BATCH
			)
		);

		$ids = is_array( $ids ) ? array_map( 'intval', $ids ) : array();

		if ( empty( $ids ) ) {
			$this->vlwp_mark_legacy_scan_completed();
			return;
		}

		$max_seen_id = $cursor;

		foreach ( $ids as $attachment_id ) {
			$max_seen_id = max( $max_seen_id, $attachment_id );

			$file = get_attached_file( $attachment_id );
			if ( empty( $file ) || ! file_exists( $file ) ) {
				continue;
			}

			if ( ! $this->vlwp_is_extension_allowed( $file ) ) {
				continue;
			}

			$this->vlwp_enqueue_attachment( $attachment_id );
		}

		update_option( self::OPTION_LEGACY_SCAN_CURSOR, $max_seen_id, false );

		if ( count( $ids ) < self::LEGACY_SCAN_BATCH ) {
			$this->vlwp_mark_legacy_scan_completed();
			return;
		}

		$this->vlwp_schedule_legacy_scan();
	}

	/**
	 * Convert one attachment immediately.
	 *
	 * @param array  $metadata Attachment metadata.
	 * @param int    $attachment_id Attachment ID.
	 * @param string $original_file Original file path.
	 * @return array
	 */
	private function vlwp_convert_attachment_now( $metadata, $attachment_id, $original_file ) {
		$converted_files = 0;
		$bytes_before = 0;
		$bytes_after = 0;

		$converted_original = $this->vlwp_convert_file_if_beneficial( $original_file );

		if ( ! empty( $converted_original['path'] ) && file_exists( $converted_original['path'] ) ) {
			$metadata['file'] = $this->vlwp_replace_extension( $metadata['file'], 'webp' );
			$metadata['mime_type'] = 'image/webp';
			$metadata['filesize'] = filesize( $converted_original['path'] );

			update_attached_file( $attachment_id, $converted_original['path'] );

			wp_update_post(
				array(
					'ID'             => $attachment_id,
					'post_mime_type' => 'image/webp',
				)
			);

			$converted_files++;
			$bytes_before += (int) $converted_original['source_size'];
			$bytes_after += (int) $converted_original['webp_size'];
		}

		if ( ! empty( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
			$upload_dir = wp_upload_dir();
			$relative_dir = trim( dirname( $metadata['file'] ), '/\\' );
			$base_dir = ( '.' === $relative_dir || '' === $relative_dir ) ? $upload_dir['basedir'] : trailingslashit( $upload_dir['basedir'] ) . $relative_dir;

			foreach ( $metadata['sizes'] as $size_key => $size_data ) {
				if ( empty( $size_data['file'] ) ) {
					continue;
				}

				$size_path = trailingslashit( $base_dir ) . $size_data['file'];
				if ( ! file_exists( $size_path ) ) {
					continue;
				}

				if ( ! $this->vlwp_is_extension_allowed( $size_path ) ) {
					continue;
				}

				$converted_size = $this->vlwp_convert_file_if_beneficial( $size_path );
				if ( empty( $converted_size['path'] ) || ! file_exists( $converted_size['path'] ) ) {
					continue;
				}

				$metadata['sizes'][ $size_key ]['file'] = $this->vlwp_replace_extension( $size_data['file'], 'webp' );
				$metadata['sizes'][ $size_key ]['mime-type'] = 'image/webp';
				$metadata['sizes'][ $size_key ]['filesize'] = filesize( $converted_size['path'] );

				$converted_files++;
				$bytes_before += (int) $converted_size['source_size'];
				$bytes_after += (int) $converted_size['webp_size'];
			}
		}

		$this->vlwp_increment_stats(
			array(
				'attachments_processed' => 1,
				'files_converted'      => $converted_files,
				'bytes_before'         => $bytes_before,
				'bytes_after'          => $bytes_after,
			)
		);

		return $metadata;
	}

	/**
	 * Prefer WebP image URL if a converted file exists.
	 *
	 * @param array|false $image Image source data.
	 * @param int         $attachment_id Attachment ID.
	 * @param string|int[] $size Requested size.
	 * @return array|false
	 */
	public function vlwp_prefer_webp_src( $image, $attachment_id, $size ) {
		if ( ! $this->vlwp_is_conversion_enabled() ) {
			return $image;
		}

		if ( ! is_array( $image ) || empty( $image[0] ) ) {
			return $image;
		}

		$webp_url = preg_replace( '/\.(jpe?g|png)$/i', '.webp', $image[0] );
		if ( $webp_url === $image[0] ) {
			return $image;
		}

		$upload_dir = wp_get_upload_dir();
		if ( empty( $upload_dir['baseurl'] ) || empty( $upload_dir['basedir'] ) ) {
			return $image;
		}

		$webp_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $webp_url );
		if ( file_exists( $webp_path ) ) {
			$image[0] = $webp_url;
		}

		return $image;
	}

	/**
	 * Get supported source types map.
	 *
	 * @return array
	 */
	public function vlwp_get_supported_type_map() {
		$types = array();
		$support_jpeg = wp_image_editor_supports( array( 'mime_type' => 'image/jpeg' ) );
		$support_png = wp_image_editor_supports( array( 'mime_type' => 'image/png' ) );

		if ( $support_jpeg ) {
			$types['jpg'] = 'JPEG (.jpg)';
			$types['jpeg'] = 'JPEG (.jpeg)';
		}

		if ( $support_png ) {
			$types['png'] = 'PNG (.png)';
		}

		return $types;
	}

	/**
	 * Get effective allowed types.
	 *
	 * @return array
	 */
	public function vlwp_get_effective_allowed_types() {
		$local_types = get_option( self::OPTION_ALLOWED_TYPES, self::DEFAULT_ALLOWED_TYPES );
		$types = is_array( $local_types ) ? $local_types : self::DEFAULT_ALLOWED_TYPES;

		if ( is_multisite() && $this->vlwp_use_network_defaults() ) {
			$network_types = get_site_option( self::NETWORK_OPTION_ALLOWED_TYPES, self::DEFAULT_ALLOWED_TYPES );
			$types = is_array( $network_types ) ? $network_types : self::DEFAULT_ALLOWED_TYPES;
		}

		$supported = array_keys( $this->vlwp_get_supported_type_map() );
		$types = array_values( array_intersect( array_map( 'sanitize_key', $types ), $supported ) );

		if ( empty( $types ) ) {
			$types = array_values( array_intersect( self::DEFAULT_ALLOWED_TYPES, $supported ) );
		}

		return array_values( array_unique( $types ) );
	}

	/**
	 * Get effective disable conversion setting.
	 *
	 * @return int
	 */
	public function vlwp_get_effective_disable_conversion() {
		$disabled = (int) get_option( self::OPTION_DISABLE_CONVERSION, 0 );

		if ( is_multisite() && $this->vlwp_use_network_defaults() ) {
			$disabled = (int) get_site_option( self::NETWORK_OPTION_DISABLE_CONVERSION, 0 );
		}

		return $disabled;
	}

	/**
	 * Get queue size.
	 *
	 * @return int
	 */
	public function vlwp_get_queue_size() {
		$queue = get_option( self::OPTION_QUEUE, array() );

		return is_array( $queue ) ? count( $queue ) : 0;
	}

	/**
	 * Get stats payload.
	 *
	 * @return array
	 */
	public function vlwp_get_stats() {
		$stats = get_option( self::OPTION_STATS, array() );
		$defaults = array(
			'attachments_processed' => 0,
			'files_converted'       => 0,
			'bytes_before'          => 0,
			'bytes_after'           => 0,
			'queued_jobs'           => 0,
			'queue_processed'       => 0,
			'queue_failed'          => 0,
			'last_updated'          => 0,
		);

		return wp_parse_args( is_array( $stats ) ? $stats : array(), $defaults );
	}

	/**
	 * Convert one file to WebP if conversion is possible.
	 *
	 * @param string $file_path Source file path.
	 * @return array|false
	 */
	private function vlwp_convert_file_if_beneficial( $file_path ) {
		if ( ! file_exists( $file_path ) ) {
			return false;
		}

		$ext = strtolower( (string) pathinfo( $file_path, PATHINFO_EXTENSION ) );
		if ( 'webp' === $ext || ! $this->vlwp_is_extension_allowed( $file_path ) ) {
			return false;
		}

		$source_size = (int) filesize( $file_path );

		$webp_path = $this->vlwp_replace_extension( $file_path, 'webp' );
		if ( empty( $webp_path ) ) {
			return false;
		}

		if ( file_exists( $webp_path ) && filesize( $webp_path ) > 0 && filemtime( $webp_path ) >= filemtime( $file_path ) ) {
			$this->vlwp_replace_with_webp( $file_path, $webp_path );
			return array(
				'path'        => $webp_path,
				'source_size' => $source_size,
				'webp_size'   => (int) filesize( $webp_path ),
			);
		}

		$editor = wp_get_image_editor( $file_path );
		if ( is_wp_error( $editor ) ) {
			return false;
		}

		if ( method_exists( $editor, 'set_quality' ) ) {
			$editor->set_quality( self::WEBP_QUALITY, 'image/webp' );
		}

		$save_result = $editor->save( $webp_path, 'image/webp' );
		if ( is_wp_error( $save_result ) || empty( $save_result['path'] ) || ! file_exists( $save_result['path'] ) ) {
			return false;
		}

		$webp_size = filesize( $save_result['path'] );
		if ( false === $webp_size || 0 >= (int) $webp_size ) {
			wp_delete_file( $save_result['path'] );
			return false;
		}

		$this->vlwp_replace_with_webp( $file_path, $save_result['path'] );

		return array(
			'path'        => $save_result['path'],
			'source_size' => $source_size,
			'webp_size'   => (int) $webp_size,
		);
	}

	/**
	 * Replace source file with generated WebP.
	 *
	 * @param string $source_path Source path.
	 * @param string $webp_path WebP path.
	 */
	private function vlwp_replace_with_webp( $source_path, $webp_path ) {
		if ( file_exists( $source_path ) ) {
			wp_delete_file( $source_path );
		}

		if ( file_exists( $webp_path ) ) {
			clearstatcache( true, $webp_path );
		}
	}

	/**
	 * Check whether conversion is enabled.
	 *
	 * @return bool
	 */
	private function vlwp_is_conversion_enabled() {
		$disabled = $this->vlwp_get_effective_disable_conversion();

		return 1 !== $disabled;
	}

	/**
	 * Check if extension is currently allowed by settings.
	 *
	 * @param string $path File path.
	 * @return bool
	 */
	private function vlwp_is_extension_allowed( $path ) {
		$ext = strtolower( (string) pathinfo( $path, PATHINFO_EXTENSION ) );
		$allowed = $this->vlwp_get_effective_allowed_types();

		return in_array( $ext, $allowed, true );
	}

	/**
	 * Check if site should inherit network defaults.
	 *
	 * @return bool
	 */
	private function vlwp_use_network_defaults() {
		if ( ! is_multisite() ) {
			return false;
		}

		return 1 === (int) get_option( self::OPTION_USE_NETWORK_DEFAULTS, 1 );
	}

	/**
	 * Check if legacy scan should run.
	 *
	 * @return bool
	 */
	private function vlwp_should_scan_legacy_uploads() {
		if ( ! $this->vlwp_is_conversion_enabled() ) {
			return false;
		}

		return 1 === (int) get_option( self::OPTION_CONVERT_EXISTING, 0 );
	}

	/**
	 * Enqueue one attachment for async conversion.
	 *
	 * @param int $attachment_id Attachment ID.
	 */
	private function vlwp_enqueue_attachment( $attachment_id ) {
		$attachment_id = (int) $attachment_id;
		if ( $attachment_id <= 0 ) {
			return;
		}

		$queue = get_option( self::OPTION_QUEUE, array() );
		$queue = is_array( $queue ) ? $queue : array();

		if ( in_array( $attachment_id, $queue, true ) ) {
			return;
		}

		$queue[] = $attachment_id;
		update_option( self::OPTION_QUEUE, array_values( array_unique( $queue ) ), false );

		$this->vlwp_increment_stats( array( 'queued_jobs' => 1 ) );
		$this->vlwp_schedule_queue_processing();
	}

	/**
	 * Schedule queue processing if needed.
	 */
	private function vlwp_schedule_queue_processing() {
		if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
			wp_schedule_single_event( time() + 30, self::CRON_HOOK );
		}
	}

	/**
	 * Maybe schedule legacy scan.
	 */
	private function vlwp_maybe_schedule_legacy_scan() {
		if ( ! $this->vlwp_should_scan_legacy_uploads() ) {
			return;
		}

		$this->vlwp_schedule_legacy_scan();
	}

	/**
	 * Schedule the next legacy scan run.
	 */
	private function vlwp_schedule_legacy_scan() {
		if ( ! wp_next_scheduled( self::LEGACY_SCAN_CRON_HOOK ) ) {
			wp_schedule_single_event( time() + self::LEGACY_SCAN_INTERVAL_SECONDS, self::LEGACY_SCAN_CRON_HOOK );
		}
	}

	/**
	 * Mark legacy scan as completed and auto-disable setting.
	 */
	private function vlwp_mark_legacy_scan_completed() {
		update_option( self::OPTION_CONVERT_EXISTING, 0, false );
		update_option( self::OPTION_LEGACY_SCAN_CURSOR, 0, false );

		$next = wp_next_scheduled( self::LEGACY_SCAN_CRON_HOOK );
		if ( false !== $next ) {
			wp_unschedule_event( $next, self::LEGACY_SCAN_CRON_HOOK );
		}
	}

	/**
	 * Increment stats payload.
	 *
	 * @param array $changes Stat deltas.
	 */
	private function vlwp_increment_stats( $changes ) {
		$stats = $this->vlwp_get_stats();

		foreach ( $changes as $key => $delta ) {
			if ( ! isset( $stats[ $key ] ) ) {
				continue;
			}

			$stats[ $key ] = (int) $stats[ $key ] + (int) $delta;
		}

		$stats['last_updated'] = time();

		update_option( self::OPTION_STATS, $stats, false );
	}

	/**
	 * Replace filename extension.
	 *
	 * @param string $path File path or relative file.
	 * @param string $new_extension New extension without dot.
	 * @return string
	 */
	private function vlwp_replace_extension( $path, $new_extension ) {
		return preg_replace( '/\.[^.]+$/', '.' . ltrim( $new_extension, '.' ), $path );
	}
}
