<?php
/**
 * Admin settings for VLWP WebP.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_WebP_Admin {
	/**
	 * Singleton instance.
	 *
	 * @var VLWP_WebP_Admin|null
	 */
	private static $instance = null;

	/**
	 * Option key for disable flag.
	 *
	 * @var string
	 */
	const OPTION_DISABLE_CONVERSION = 'vlwp_webp_disable_conversion';

	/**
	 * Option key for local allowed image types.
	 *
	 * @var string
	 */
	const OPTION_ALLOWED_TYPES = 'vlwp_webp_allowed_types';

	/**
	 * Option key for background conversion of previously uploaded images.
	 *
	 * @var string
	 */
	const OPTION_CONVERT_EXISTING = 'vlwp_webp_convert_existing';

	/**
	 * Option key for per-site network override toggle.
	 *
	 * @var string
	 */
	const OPTION_USE_NETWORK_DEFAULTS = 'vlwp_webp_use_network_defaults';

	/**
	 * Site option key for network disable default.
	 *
	 * @var string
	 */
	const NETWORK_OPTION_DISABLE_CONVERSION = 'vlwp_webp_network_disable_conversion';

	/**
	 * Site option key for network allowed types default.
	 *
	 * @var string
	 */
	const NETWORK_OPTION_ALLOWED_TYPES = 'vlwp_webp_network_allowed_types';

	/**
	 * Settings section slug.
	 *
	 * @var string
	 */
	const SETTINGS_SECTION = 'vlwp_webp_settings_section';

	/**
	 * Get singleton instance.
	 *
	 * @return VLWP_WebP_Admin
	 */
	public static function vlwp_get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Init admin hooks.
	 */
	public function vlwp_init() {
		add_action( 'admin_init', array( $this, 'vlwp_register_setting' ) );
		add_action( 'admin_init', array( $this, 'vlwp_add_settings_section_and_fields' ) );
		add_filter( 'all_plugins', array( $this, 'vlwp_translate_plugin_description' ) );

		if ( is_multisite() ) {
			add_action( 'network_admin_menu', array( $this, 'vlwp_register_network_menu' ) );
			add_action( 'network_admin_edit_vlwp_webp_save_network_settings', array( $this, 'vlwp_handle_network_settings_save' ) );
		}
	}

	/**
	 * Activation setup.
	 */
	public static function vlwp_activate() {
		if ( false === get_option( self::OPTION_DISABLE_CONVERSION, false ) ) {
			update_option( self::OPTION_DISABLE_CONVERSION, 0 );
		}

		if ( false === get_option( self::OPTION_ALLOWED_TYPES, false ) ) {
			update_option( self::OPTION_ALLOWED_TYPES, array( 'jpg', 'jpeg', 'png' ) );
		}

		if ( false === get_option( self::OPTION_CONVERT_EXISTING, false ) ) {
			update_option( self::OPTION_CONVERT_EXISTING, 0 );
		}

		if ( is_multisite() ) {
			if ( false === get_option( self::OPTION_USE_NETWORK_DEFAULTS, false ) ) {
				update_option( self::OPTION_USE_NETWORK_DEFAULTS, 1 );
			}

			if ( false === get_site_option( self::NETWORK_OPTION_DISABLE_CONVERSION, false ) ) {
				update_site_option( self::NETWORK_OPTION_DISABLE_CONVERSION, 0 );
			}

			if ( false === get_site_option( self::NETWORK_OPTION_ALLOWED_TYPES, false ) ) {
				update_site_option( self::NETWORK_OPTION_ALLOWED_TYPES, array( 'jpg', 'jpeg', 'png' ) );
			}
		}
	}

	/**
	 * Register media setting.
	 */
	public function vlwp_register_setting() {
		register_setting( 'media', self::OPTION_DISABLE_CONVERSION, array( $this, 'vlwp_sanitize_disable_flag' ) );
		register_setting( 'media', self::OPTION_ALLOWED_TYPES, array( $this, 'vlwp_sanitize_allowed_types' ) );
		register_setting( 'media', self::OPTION_CONVERT_EXISTING, array( $this, 'vlwp_sanitize_disable_flag' ) );

		if ( is_multisite() ) {
			register_setting( 'media', self::OPTION_USE_NETWORK_DEFAULTS, array( $this, 'vlwp_sanitize_disable_flag' ) );
		}

		if ( false === get_option( self::OPTION_DISABLE_CONVERSION, false ) ) {
			update_option( self::OPTION_DISABLE_CONVERSION, 0 );
		}

		if ( false === get_option( self::OPTION_ALLOWED_TYPES, false ) ) {
			update_option( self::OPTION_ALLOWED_TYPES, array( 'jpg', 'jpeg', 'png' ) );
		}

		if ( false === get_option( self::OPTION_CONVERT_EXISTING, false ) ) {
			update_option( self::OPTION_CONVERT_EXISTING, 0 );
		}

		if ( is_multisite() && false === get_option( self::OPTION_USE_NETWORK_DEFAULTS, false ) ) {
			update_option( self::OPTION_USE_NETWORK_DEFAULTS, 1 );
		}
	}

	/**
	 * Add media settings section and fields.
	 */
	public function vlwp_add_settings_section_and_fields() {
		add_settings_section(
			self::SETTINGS_SECTION,
			__( 'WebP', 'vlwp-webp' ),
			array( $this, 'vlwp_render_section_description' ),
			'media'
		);

		if ( is_multisite() ) {
			add_settings_field(
				self::OPTION_USE_NETWORK_DEFAULTS,
				__( 'Use network defaults', 'vlwp-webp' ),
				array( $this, 'vlwp_render_network_override_field' ),
				'media',
				self::SETTINGS_SECTION
			);
		}

		add_settings_field(
			self::OPTION_DISABLE_CONVERSION,
			__( 'Conversion status', 'vlwp-webp' ),
			array( $this, 'vlwp_render_checkbox_field' ),
			'media',
			self::SETTINGS_SECTION
		);

		add_settings_field(
			self::OPTION_ALLOWED_TYPES,
			__( 'Convertible file types', 'vlwp-webp' ),
			array( $this, 'vlwp_render_type_multiselect_field' ),
			'media',
			self::SETTINGS_SECTION
		);

		add_settings_field(
			self::OPTION_CONVERT_EXISTING,
			__( 'Previously uploaded images', 'vlwp-webp' ),
			array( $this, 'vlwp_render_convert_existing_field' ),
			'media',
			self::SETTINGS_SECTION
		);

		add_settings_field(
			'vlwp_webp_stats_dashboard',
			__( 'Conversion stats', 'vlwp-webp' ),
			array( $this, 'vlwp_render_stats_dashboard' ),
			'media',
			self::SETTINGS_SECTION
		);
	}

	/**
	 * Sanitize disable option.
	 *
	 * @param mixed $value Raw value.
	 * @return int
	 */
	public function vlwp_sanitize_disable_flag( $value ) {
		return ! empty( $value ) ? 1 : 0;
	}

	/**
	 * Sanitize selected file types.
	 *
	 * @param mixed $value Raw value.
	 * @return array
	 */
	public function vlwp_sanitize_allowed_types( $value ) {
		$types = is_array( $value ) ? $value : array();
		$types = array_map( 'sanitize_key', $types );
		$types = array_filter( $types );

		$supported_types = array_keys( VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_supported_type_map() );
		$clean_types = array_values( array_intersect( $types, $supported_types ) );

		if ( empty( $clean_types ) ) {
			$clean_types = array_values( array_intersect( array( 'jpg', 'jpeg', 'png' ), $supported_types ) );
		}

		return array_values( array_unique( $clean_types ) );
	}

	/**
	 * Render section description.
	 */
	public function vlwp_render_section_description() {
		echo '<p>' . esc_html__( 'Configure automatic WebP conversion behavior.', 'vlwp-webp' ) . '</p>';
	}

	/**
	 * Render network override field.
	 */
	public function vlwp_render_network_override_field() {
		$value = (int) get_option( self::OPTION_USE_NETWORK_DEFAULTS, 1 );
		?>
		<input type="checkbox" id="<?php echo esc_attr( self::OPTION_USE_NETWORK_DEFAULTS ); ?>" name="<?php echo esc_attr( self::OPTION_USE_NETWORK_DEFAULTS ); ?>" value="1" <?php checked( 1, $value ); ?> />
		<label for="<?php echo esc_attr( self::OPTION_USE_NETWORK_DEFAULTS ); ?>"><?php esc_html_e( 'Use network-wide defaults for this site', 'vlwp-webp' ); ?></label>
		<p class="description"><?php esc_html_e( 'If enabled, local settings below are ignored and network values are applied.', 'vlwp-webp' ); ?></p>
		<?php
	}

	/**
	 * Render disable conversion checkbox.
	 */
	public function vlwp_render_checkbox_field() {
		$value = (int) get_option( self::OPTION_DISABLE_CONVERSION, 0 );
		$effective_disabled = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_effective_disable_conversion();
		?>
		<input type="checkbox" id="<?php echo esc_attr( self::OPTION_DISABLE_CONVERSION ); ?>" name="<?php echo esc_attr( self::OPTION_DISABLE_CONVERSION ); ?>" value="1" <?php checked( 1, $value ); ?> />
		<label for="<?php echo esc_attr( self::OPTION_DISABLE_CONVERSION ); ?>"><?php esc_html_e( 'Disable automatic conversion', 'vlwp-webp' ); ?></label>
		<p class="description"><?php esc_html_e( 'When enabled, uploads stay unchanged.', 'vlwp-webp' ); ?></p>
		<p class="description"><?php echo esc_html( sprintf( __( 'Effective status: %s', 'vlwp-webp' ), $effective_disabled ? __( 'Disabled', 'vlwp-webp' ) : __( 'Enabled', 'vlwp-webp' ) ) ); ?></p>
		<?php
	}

	/**
	 * Render multiselect for allowed source types.
	 */
	public function vlwp_render_type_multiselect_field() {
		$selected = get_option( self::OPTION_ALLOWED_TYPES, array( 'jpg', 'jpeg', 'png' ) );
		$selected = is_array( $selected ) ? array_map( 'sanitize_key', $selected ) : array();
		$supported = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_supported_type_map();
		$effective_types = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_effective_allowed_types();
		?>
		<select id="<?php echo esc_attr( self::OPTION_ALLOWED_TYPES ); ?>" name="<?php echo esc_attr( self::OPTION_ALLOWED_TYPES ); ?>[]" multiple="multiple" size="4" style="min-width:220px;">
			<?php foreach ( $supported as $ext => $label ) : ?>
				<option value="<?php echo esc_attr( $ext ); ?>" <?php selected( in_array( $ext, $selected, true ) ); ?>><?php echo esc_html( $label ); ?></option>
			<?php endforeach; ?>
		</select>
		<p class="description"><?php esc_html_e( 'Only supported image types are shown. Hold Ctrl/Cmd to select multiple entries.', 'vlwp-webp' ); ?></p>
		<p class="description"><?php echo esc_html( sprintf( __( 'Effective types: %s', 'vlwp-webp' ), implode( ', ', $effective_types ) ) ); ?></p>
		<?php
	}

	/**
	 * Render checkbox for converting existing uploads.
	 */
	public function vlwp_render_convert_existing_field() {
		$enabled = (int) get_option( self::OPTION_CONVERT_EXISTING, 0 );
		?>
		<input type="checkbox" id="<?php echo esc_attr( self::OPTION_CONVERT_EXISTING ); ?>" name="<?php echo esc_attr( self::OPTION_CONVERT_EXISTING ); ?>" value="1" <?php checked( 1, $enabled ); ?> />
		<label for="<?php echo esc_attr( self::OPTION_CONVERT_EXISTING ); ?>"><?php esc_html_e( 'Also convert previously uploaded images', 'vlwp-webp' ); ?></label>
		<p class="description"><?php esc_html_e( 'Scans older uploads in very small background batches and adds eligible files to the queue.', 'vlwp-webp' ); ?></p>
		<p class="description"><?php esc_html_e( 'This option is automatically disabled after the full scan is finished.', 'vlwp-webp' ); ?></p>
		<?php
	}

	/**
	 * Render stats dashboard.
	 */
	public function vlwp_render_stats_dashboard() {
		$stats = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_stats();
		$queue_size = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_queue_size();
		$saved_bytes = max( 0, (int) $stats['bytes_before'] - (int) $stats['bytes_after'] );
		$saved_mb = $saved_bytes / 1024 / 1024;
		?>
		<div>
			<p><strong><?php esc_html_e( 'Converted files:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( (int) $stats['files_converted'] ) ); ?></p>
			<p><strong><?php esc_html_e( 'Processed attachments:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( (int) $stats['attachments_processed'] ) ); ?></p>
			<p><strong><?php esc_html_e( 'Queue size:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( $queue_size ) ); ?></p>
			<p><strong><?php esc_html_e( 'Queued jobs:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( (int) $stats['queued_jobs'] ) ); ?></p>
			<p><strong><?php esc_html_e( 'Queue processed jobs:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( (int) $stats['queue_processed'] ) ); ?></p>
			<p><strong><?php esc_html_e( 'Queue failed jobs:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( (int) $stats['queue_failed'] ) ); ?></p>
			<p><strong><?php esc_html_e( 'Estimated MB saved:', 'vlwp-webp' ); ?></strong> <?php echo esc_html( number_format_i18n( $saved_mb, 2 ) . ' MB' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Register network settings page.
	 */
	public function vlwp_register_network_menu() {
		add_submenu_page(
			'settings.php',
			__( 'WebP', 'vlwp-webp' ),
			__( 'WebP', 'vlwp-webp' ),
			'manage_network_options',
			'vlwp-webp-network',
			array( $this, 'vlwp_render_network_settings_page' )
		);
	}

	/**
	 * Render network settings page.
	 */
	public function vlwp_render_network_settings_page() {
		if ( ! current_user_can( 'manage_network_options' ) ) {
			return;
		}

		$disable_conversion = (int) get_site_option( self::NETWORK_OPTION_DISABLE_CONVERSION, 0 );
		$selected_types = get_site_option( self::NETWORK_OPTION_ALLOWED_TYPES, array( 'jpg', 'jpeg', 'png' ) );
		$selected_types = is_array( $selected_types ) ? $selected_types : array();
		$supported = VLWP_WebP_Core::vlwp_get_instance()->vlwp_get_supported_type_map();
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'WebP Network Defaults', 'vlwp-webp' ); ?></h1>
			<form method="post" action="<?php echo esc_url( network_admin_url( 'edit.php?action=vlwp_webp_save_network_settings' ) ); ?>">
				<?php wp_nonce_field( 'vlwp_webp_save_network_settings', 'vlwp_webp_network_nonce' ); ?>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php esc_html_e( 'Conversion status', 'vlwp-webp' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="<?php echo esc_attr( self::NETWORK_OPTION_DISABLE_CONVERSION ); ?>" value="1" <?php checked( 1, $disable_conversion ); ?> />
								<?php esc_html_e( 'Disable automatic conversion network-wide', 'vlwp-webp' ); ?>
							</label>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Convertible file types', 'vlwp-webp' ); ?></th>
						<td>
							<select name="<?php echo esc_attr( self::NETWORK_OPTION_ALLOWED_TYPES ); ?>[]" multiple="multiple" size="4" style="min-width:220px;">
								<?php foreach ( $supported as $ext => $label ) : ?>
									<option value="<?php echo esc_attr( $ext ); ?>" <?php selected( in_array( $ext, $selected_types, true ) ); ?>><?php echo esc_html( $label ); ?></option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php esc_html_e( 'Sites can use these defaults unless they disable network defaults on their own Media settings page.', 'vlwp-webp' ); ?></p>
						</td>
					</tr>
				</table>
				<?php submit_button( __( 'Save Settings', 'vlwp-webp' ) ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle network settings save.
	 */
	public function vlwp_handle_network_settings_save() {
		if ( ! current_user_can( 'manage_network_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'vlwp-webp' ) );
		}

		check_admin_referer( 'vlwp_webp_save_network_settings', 'vlwp_webp_network_nonce' );

		$disable = isset( $_POST[ self::NETWORK_OPTION_DISABLE_CONVERSION ] ) ? $this->vlwp_sanitize_disable_flag( wp_unslash( $_POST[ self::NETWORK_OPTION_DISABLE_CONVERSION ] ) ) : 0;
		$types = isset( $_POST[ self::NETWORK_OPTION_ALLOWED_TYPES ] ) ? $this->vlwp_sanitize_allowed_types( wp_unslash( $_POST[ self::NETWORK_OPTION_ALLOWED_TYPES ] ) ) : array();

		update_site_option( self::NETWORK_OPTION_DISABLE_CONVERSION, $disable );
		update_site_option( self::NETWORK_OPTION_ALLOWED_TYPES, $types );

		wp_safe_redirect( network_admin_url( 'settings.php?page=vlwp-webp-network' ) );
		exit;
	}

	/**
	 * Translate plugin description in the plugins list.
	 *
	 * @param array $plugins All plugins.
	 * @return array
	 */
	public function vlwp_translate_plugin_description( $plugins ) {
		$plugin_basename = plugin_basename( vlwp_webp_plugin_file );

		if ( isset( $plugins[ $plugin_basename ] ) ) {
			$plugins[ $plugin_basename ]['Description'] = __( 'Automatically converts uploaded JPEG/PNG images to WebP with an on/off setting.', 'vlwp-webp' );
		}

		return $plugins;
	}
}
