<?php
/**
 * Core update logic.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Updater_Core {
	/**
	 * Singleton instance.
	 *
	 * @var VLWP_Updater_Core|null
	 */
	private static $instance = null;

	/**
	 * Repository URL.
	 *
	 * @var string
	 */
	private $repo_url = '';

	/**
	 * Get singleton.
	 *
	 * @return VLWP_Updater_Core
	 */
	public static function vlwp_get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Init hooks.
	 */
	public function vlwp_init() {
		$this->repo_url = apply_filters( 'vlwp_updater_repository_url', 'https://wp.vitali-lutz.de/plugins/' );

		if ( empty( $this->repo_url ) ) {
			return;
		}

		add_action( 'vlwp_check_updates', array( $this, 'vlwp_check_for_updates' ) );

		if ( ! wp_next_scheduled( 'vlwp_check_updates' ) ) {
			wp_schedule_event( time(), 'weekly', 'vlwp_check_updates' );
		}
	}

	/**
	 * Check for plugin updates and install if available.
	 */
	public function vlwp_check_for_updates() {
		if ( empty( $this->repo_url ) ) {
			return;
		}

		$vlwp_plugins = $this->vlwp_get_vlwp_plugins();
		if ( empty( $vlwp_plugins ) ) {
			return;
		}

		foreach ( $vlwp_plugins as $plugin_slug => $plugin_data ) {
			$this->vlwp_update_plugin_if_available( $plugin_slug, $plugin_data );
		}
	}

	/**
	 * Get all installed VLWP plugins.
	 *
	 * @return array
	 */
	private function vlwp_get_vlwp_plugins() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins = get_plugins();
		$vlwp_plugins = array();

		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			if ( 0 === strpos( basename( dirname( $plugin_file ) ), 'vlwp-' ) ) {
				$slug = basename( dirname( $plugin_file ) );
				$version = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '0.0.0';
				$vlwp_plugins[ $slug ] = array(
					'file'    => $plugin_file,
					'version' => $version,
				);
			}
		}

		return $vlwp_plugins;
	}

	/**
	 * Update a plugin if a newer version is available.
	 *
	 * @param string $plugin_slug Plugin slug.
	 * @param array  $plugin_data Plugin data.
	 */
	private function vlwp_update_plugin_if_available( $plugin_slug, $plugin_data ) {
		$remote_info = $this->vlwp_fetch_plugin_info( $plugin_slug );
		if ( empty( $remote_info ) ) {
			return;
		}

		$current_version = $plugin_data['version'];
		$remote_version = isset( $remote_info['version'] ) ? $remote_info['version'] : '';
		$download_url = isset( $remote_info['download_url'] ) ? $remote_info['download_url'] : '';

		if ( empty( $remote_version ) || empty( $download_url ) ) {
			return;
		}

		if ( version_compare( $current_version, $remote_version, '<' ) ) {
			$this->vlwp_install_plugin_update( $plugin_slug, $download_url, $plugin_data );
		}
	}

	/**
	 * Fetch plugin info from remote repository.
	 *
	 * @param string $plugin_slug Plugin slug.
	 * @return array
	 */
	private function vlwp_fetch_plugin_info( $plugin_slug ) {
		$url = trailingslashit( $this->repo_url ) . $plugin_slug;

		$response = wp_remote_get( $url, array( 'timeout' => 10 ) );
		if ( is_wp_error( $response ) ) {
			return array();
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		return is_array( $data ) ? $data : array();
	}

	/**
	 * Download and install plugin update.
	 *
	 * @param string $plugin_slug Plugin slug.
	 * @param string $download_url Download URL.
	 * @param array  $plugin_data Plugin data.
	 */
	private function vlwp_install_plugin_update( $plugin_slug, $download_url, $plugin_data ) {
		if ( ! class_exists( 'WP_Filesystem_Direct' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		WP_Filesystem();
		global $wp_filesystem;

		$plugins_dir = WP_PLUGIN_DIR;
		$plugin_dir = trailingslashit( $plugins_dir ) . $plugin_slug;

		$tmp_dir = trailingslashit( get_temp_dir() ) . 'vlwp-update-' . uniqid();
		wp_mkdir_p( $tmp_dir );

		$zip_file = trailingslashit( $tmp_dir ) . 'plugin.zip';

		$response = wp_remote_get( $download_url, array( 'timeout' => 30 ) );
		if ( is_wp_error( $response ) ) {
			$this->vlwp_cleanup_temp_dir( $tmp_dir );
			return;
		}

		$zip_data = wp_remote_retrieve_body( $response );
		if ( empty( $zip_data ) ) {
			$this->vlwp_cleanup_temp_dir( $tmp_dir );
			return;
		}

		$wp_filesystem->put_contents( $zip_file, $zip_data );

		if ( ! class_exists( 'ZipArchive' ) ) {
			$this->vlwp_cleanup_temp_dir( $tmp_dir );
			return;
		}

		$zip = new ZipArchive();
		if ( true !== $zip->open( $zip_file ) ) {
			$this->vlwp_cleanup_temp_dir( $tmp_dir );
			return;
		}

		$extract_dir = trailingslashit( $tmp_dir ) . 'extracted';
		$zip->extractTo( $extract_dir );
		$zip->close();

		$extracted_contents = array_diff( scandir( $extract_dir ), array( '.', '..' ) );
		if ( empty( $extracted_contents ) ) {
			$this->vlwp_cleanup_temp_dir( $tmp_dir );
			return;
		}

		$source_dir = trailingslashit( $extract_dir ) . reset( $extracted_contents );

		if ( is_dir( $plugin_dir ) ) {
			$this->vlwp_remove_dir_recursively( $plugin_dir );
		}

		wp_mkdir_p( $plugin_dir );

		$this->vlwp_copy_dir_recursively( $source_dir, $plugin_dir );

		$this->vlwp_cleanup_temp_dir( $tmp_dir );

		$this->vlwp_run_update_script( $plugin_slug, $plugin_dir );

		do_action( 'vlwp_plugin_updated', $plugin_slug );
	}

	/**
	 * Run update.php if it exists.
	 *
	 * @param string $plugin_slug Plugin slug.
	 * @param string $plugin_dir Plugin directory.
	 */
	private function vlwp_run_update_script( $plugin_slug, $plugin_dir ) {
		$update_file = trailingslashit( $plugin_dir ) . 'update.php';

		if ( ! file_exists( $update_file ) ) {
			return;
		}

		require_once ABSPATH . 'wp-load.php';
		require $update_file;
	}

	/**
	 * Copy directory recursively.
	 *
	 * @param string $source Source directory.
	 * @param string $dest Destination directory.
	 */
	private function vlwp_copy_dir_recursively( $source, $dest ) {
		global $wp_filesystem;

		if ( ! is_dir( $source ) ) {
			return;
		}

		if ( ! is_dir( $dest ) ) {
			wp_mkdir_p( $dest );
		}

		$files = scandir( $source );
		if ( false === $files ) {
			return;
		}

		foreach ( $files as $file ) {
			if ( '.' === $file || '..' === $file ) {
				continue;
			}

			$source_path = trailingslashit( $source ) . $file;
			$dest_path = trailingslashit( $dest ) . $file;

			if ( is_dir( $source_path ) ) {
				$this->vlwp_copy_dir_recursively( $source_path, $dest_path );
			} else {
				$content = $wp_filesystem->get_contents( $source_path );
				if ( false !== $content ) {
					$wp_filesystem->put_contents( $dest_path, $content );
				}
			}
		}
	}

	/**
	 * Remove directory recursively.
	 *
	 * @param string $dir Directory.
	 */
	private function vlwp_remove_dir_recursively( $dir ) {
		global $wp_filesystem;

		if ( ! is_dir( $dir ) ) {
			return;
		}

		$files = scandir( $dir );
		if ( false === $files ) {
			return;
		}

		foreach ( $files as $file ) {
			if ( '.' === $file || '..' === $file ) {
				continue;
			}

			$path = trailingslashit( $dir ) . $file;

			if ( is_dir( $path ) ) {
				$this->vlwp_remove_dir_recursively( $path );
			} else {
				$wp_filesystem->delete( $path );
			}
		}

		@rmdir( $dir );
	}

	/**
	 * Cleanup temporary directory.
	 *
	 * @param string $dir Directory.
	 */
	private function vlwp_cleanup_temp_dir( $dir ) {
		if ( ! is_dir( $dir ) ) {
			return;
		}

		$files = scandir( $dir );
		if ( false === $files ) {
			return;
		}

		foreach ( $files as $file ) {
			if ( '.' === $file || '..' === $file ) {
				continue;
			}

			$path = trailingslashit( $dir ) . $file;

			if ( is_dir( $path ) ) {
				$this->vlwp_cleanup_temp_dir( $path );
			} else {
				@unlink( $path );
			}
		}

		@rmdir( $dir );
	}
}
