<?php
defined( 'ABSPATH' ) || exit;

class VLWP_SEO_Core {

    public static function init() {
        // Register post meta
        add_action( 'init', [ __CLASS__, 'register_post_meta' ] );

        // Front-end head output
        add_action( 'wp_head', [ __CLASS__, 'output_meta_tags' ], 1 );

        // Ensure canonical tag exists
        add_action( 'template_redirect', [ __CLASS__, 'maybe_buffer_for_canonical' ], 1 );

        // Enqueue editor script
        add_action( 'enqueue_block_editor_assets', [ __CLASS__, 'enqueue_editor_assets' ] );

        // REST routes for direct AJAX meta save/load
        add_action( 'rest_api_init', [ __CLASS__, 'register_rest_routes' ] );
    }

    public static function register_rest_routes() {
        register_rest_route( 'vlwp-seo/v1', '/meta/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [ __CLASS__, 'rest_get_meta' ],
            'permission_callback' => function( $request ) {
                $post_id = (int) $request['id'];
                return current_user_can( 'edit_post', $post_id );
            },
        ] );

        register_rest_route( 'vlwp-seo/v1', '/meta/(?P<id>\d+)', [
            'methods' => 'POST',
            'callback' => [ __CLASS__, 'rest_save_meta' ],
            'permission_callback' => function( $request ) {
                $post_id = (int) $request['id'];
                return current_user_can( 'edit_post', $post_id );
            },
            'args' => [],
        ] );

        register_rest_route( 'vlwp-seo/v1', '/score/(?P<id>\d+)', [
            'methods' => [ 'GET', 'POST' ],
            'callback' => [ __CLASS__, 'rest_score_keyword' ],
            'permission_callback' => function( $request ) {
                $post_id = (int) $request['id'];
                return current_user_can( 'edit_post', $post_id );
            },
            'args' => [
                'keyword' => [ 'required' => true ],
                'content' => [],
                'title' => [],
                'post_title' => [],
                'meta_title' => [],
                'meta_desc' => [],
            ],
        ] );
    }

    public static function rest_score_keyword( $request ) {
        $post_id = (int) $request->get_param( 'id' );
        $keyword = trim( (string) $request->get_param( 'keyword' ) );
        $content = $request->get_param( 'content' );
        $title = $request->get_param( 'title' );
        $post_title = $request->get_param( 'post_title' );
        $meta_title = $request->get_param( 'meta_title' );
        $meta_desc = $request->get_param( 'meta_desc' );

        if ( empty( $keyword ) ) {
            return new WP_Error( 'no_keyword', 'Keyword required', [ 'status' => 400 ] );
        }

        $options = [];
        if ( ! is_null( $content ) ) {
            $options['content_override'] = $content;
        }
        if ( ! is_null( $title ) ) {
            $options['title_override'] = $title;
        }
        if ( ! is_null( $post_title ) ) {
            $options['post_title_override'] = $post_title;
        }
        if ( ! is_null( $meta_title ) ) {
            $options['meta_title_override'] = $meta_title;
        }
        if ( ! is_null( $meta_desc ) ) {
            $options['meta_desc_override'] = $meta_desc;
        }

        // Run compute in try/catch to surface errors into REST response where possible
        try {
            error_log( 'vlwp-seo: score request for post ' . $post_id . ' keyword [' . $keyword . ']' );
            $result = self::compute_seo_score( $post_id, $keyword, $options );
            if ( is_wp_error( $result ) ) {
                return rest_ensure_response( [ 'error' => $result->get_error_message() ] );
            }
            return rest_ensure_response( $result );
        } catch ( Throwable $e ) {
            error_log( 'vlwp-seo: compute_seo_score threw: ' . $e->getMessage() );
            $data = [ 'error' => 'compute_exception', 'message' => $e->getMessage(), 'trace' => $e->getTraceAsString() ];
            return new WP_Error( 'compute_exception', $e->getMessage(), [ 'status' => 500, 'details' => $data ] );
        }
    }

    /**
     * Compute a lightweight SEO score for a single keyword for a given post.
     * Returns breakdown and final score (0-100).
     */
    public static function compute_seo_score( $post_id, $keyword, $options = [] ) {
        $post = get_post( $post_id );
        if ( ! $post ) {
            return [ 'error' => 'post_not_found', 'score' => 0 ];
        }

        // Configuration / defaults
        $min_word_count = isset( $options['min_word_count'] ) ? (int) $options['min_word_count'] : 300;

        // Gather fields (allow overrides from $options for live editor scoring)
        $meta_title = get_post_meta( $post_id, 'vlwp_seo_meta_title', true );
        $post_title = get_the_title( $post_id );
        $meta_desc = get_post_meta( $post_id, 'vlwp_seo_meta_description', true );
        $slug = $post->post_name;
        $content = $post->post_content ?: '';
        // Apply overrides if provided (from editor live content)
        if ( isset( $options['content_override'] ) ) {
            $content = (string) $options['content_override'];
        }
        if ( isset( $options['title_override'] ) && $options['title_override'] !== '' ) {
            // legacy/title_override: apply to both post and meta title for compatibility
            $meta_title = (string) $options['title_override'];
            $post_title = (string) $options['title_override'];
        }
        if ( isset( $options['post_title_override'] ) && $options['post_title_override'] !== '' ) {
            $post_title = (string) $options['post_title_override'];
        }
        if ( isset( $options['meta_title_override'] ) && $options['meta_title_override'] !== '' ) {
            $meta_title = (string) $options['meta_title_override'];
        }
        if ( isset( $options['meta_desc_override'] ) && $options['meta_desc_override'] !== '' ) {
            $meta_desc = (string) $options['meta_desc_override'];
        }
        $plain = wp_strip_all_tags( $content );
        $words = str_word_count( $plain );

        // Headings
        preg_match( '/<h1[^>]*>(.*?)<\/h1>/is', $content, $m1 );
        $h1 = isset( $m1[1] ) ? wp_strip_all_tags( $m1[1] ) : '';
        preg_match_all( '/<h[2-3][^>]*>(.*?)<\/h[2-3]>/is', $content, $m23 );
        $h2s = isset( $m23[1] ) ? $m23[1] : [];

        // Images
        preg_match_all( '/<img[^>]+>/i', $content, $img_tags );
        $images = [];
        if ( ! empty( $img_tags[0] ) ) {
            foreach ( $img_tags[0] as $tag ) {
                preg_match( '/alt=["\']([^"\']+)["\']/i', $tag, $alt );
                preg_match( '/src=["\']([^"\']+)["\']/i', $tag, $src );
                $src_val = isset( $src[1] ) ? $src[1] : '';
                $filename = basename( parse_url( $src_val, PHP_URL_PATH ) );
                $images[] = [ 'alt' => isset( $alt[1] ) ? $alt[1] : '', 'filename' => $filename ];
            }
        }

        // Links
        preg_match_all( '/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $content, $links );
        $internal = 0;
        $external = 0;
        $home_host = parse_url( home_url(), PHP_URL_HOST );
        if ( ! empty( $links[1] ) ) {
            foreach ( $links[1] as $l ) {
                $host = parse_url( $l, PHP_URL_HOST );
                if ( ! $host || $host === $home_host ) {
                    $internal++;
                } else {
                    $external++;
                }
            }
        }

        // link_relevance_bonus will be computed after helper closures are available

        // Technical checks
        $noindex = get_post_meta( $post_id, 'vlwp_seo_noindex', true );
        $canonical = get_post_meta( $post_id, 'vlwp_seo_canonical', true );
        $has_canonical = ! empty( $canonical );
        $has_schema = ( false !== strpos( $content, 'application/ld+json' ) || false !== strpos( $content, 'schema.org' ) );

        // Keywords
        $all_keywords = get_post_meta( $post_id, 'vlwp_seo_keywords', true );
        $kw_list = array_filter( array_map( 'trim', explode( ',', (string) $all_keywords ) ) );
        // Primary is $keyword passed; secondaries are other keywords
        $secondaries = array_values( array_filter( $kw_list, function( $k ) use ( $keyword ) { return mb_strtolower( $k ) !== mb_strtolower( $keyword ); } ) );

        // Normalization helpers
        $normalize = function( $s ) {
            $s = mb_strtolower( trim( strip_tags( $s ) ) );
            $s = preg_replace( '/[^\p{L}\p{N}\s]+/u', ' ', $s );
            $s = preg_replace( '/\s+/u', ' ', $s );
            return $s;
        };

        $stem = function( $w ) {
            $w = mb_strtolower( $w );
            $suffixes = [ 'ing', 'ed', 'es', 's', 'ion', 'ions', 'ity', 'ies' ];
            foreach ( $suffixes as $suf ) {
                if ( mb_substr( $w, -mb_strlen( $suf ) ) === $suf && mb_strlen( $w ) > mb_strlen( $suf ) + 2 ) {
                    $w = mb_substr( $w, 0, mb_strlen( $w ) - mb_strlen( $suf ) );
                    break;
                }
            }
            return $w;
        };

        $contains = function( $hay, $needle ) use ( $normalize, $stem ) {
            if ( empty( $needle ) || empty( $hay ) ) return false;
            $hay_norm = $normalize( $hay );
            $needle_norm = $normalize( $needle );
            // exact word match
            if ( preg_match( '/\b' . preg_quote( $needle_norm, '/' ) . '\b/ui', $hay_norm ) ) return true;
            // stem match
            $needle_stem = $stem( $needle_norm );
            $words = preg_split( '/\s+/u', $hay_norm );
            foreach ( $words as $w ) {
                if ( $stem( $w ) === $needle_stem ) return true;
            }
            return false;
        };

        // Sanity check for closures - log if any are unexpectedly not callable
        if ( ! is_callable( $contains ) || ! is_callable( $stem ) || ! is_callable( $normalize ) ) {
            error_log( 'vlwp-seo: closure check - contains:' . ( is_callable( $contains ) ? 'ok' : 'not_callable' ) . ' stem:' . ( is_callable( $stem ) ? 'ok' : 'not_callable' ) . ' normalize:' . ( is_callable( $normalize ) ? 'ok' : 'not_callable' ) );
        }

        // Link relevance: reward links whose anchor text or URL contains the keyword
        $link_relevance_bonus = 0;
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>(.*?)<\/a>/is', $content, $link_matches);
        if ( ! empty( $link_matches[0] ) ) {
            foreach ( $link_matches[0] as $i => $full_tag ) {
                $href_val = isset( $link_matches[1][$i] ) ? $link_matches[1][$i] : '';
                $anchor_text = isset( $link_matches[2][$i] ) ? wp_strip_all_tags( $link_matches[2][$i] ) : '';
                if ( $contains( $anchor_text, $keyword ) || $contains( $href_val, $keyword ) ) {
                    $link_relevance_bonus += 3; // small bonus per relevant link
                }
            }
            // cap bonus to avoid runaway influence
            if ( $link_relevance_bonus > 6 ) $link_relevance_bonus = 6;
        }

        // Occurrences in body (for density)
        $plain_norm = $normalize( $plain );
        $primary_occurrences = 0;
        if ( $plain_norm !== '' ) {
            $pattern = '/\b' . preg_quote( $normalize( $keyword ), '/' ) . '\b/iu';
            $primary_occurrences = preg_match_all( $pattern, $plain_norm );
        }
        $density = $words > 0 ? ( $primary_occurrences / $words ) * 100 : 0;

        // 1) Keyword Placement (40%) weights
        $placement_weights = [
            'title' => 12, // split between post_title and meta_title
            'meta_title' => 4, // additional explicit weight for meta title
            'h1' => 8,
            'first100' => 8,
            'h2' => 5,
            'meta' => 6, // meta description
            'slug' => 3,
            'body' => 2,
        ];
        $placement_score_raw = 0;
        $title_weight = $placement_weights['title'];
        $half_title = $title_weight / 2;
        if ( $contains( $post_title, $keyword ) ) $placement_score_raw += $half_title;
        if ( $contains( $meta_title, $keyword ) ) $placement_score_raw += $half_title;
        // additional explicit meta_title weight
        if ( $contains( $meta_title, $keyword ) ) $placement_score_raw += $placement_weights['meta_title'];
        if ( $contains( $h1, $keyword ) ) $placement_score_raw += $placement_weights['h1'];
        $first100 = implode( ' ', array_slice( preg_split( '/\s+/u', $plain_norm ), 0, 100 ) );
        if ( $contains( $first100, $keyword ) ) $placement_score_raw += $placement_weights['first100'];
        if ( ! empty( $h2s ) ) {
            $found_h2 = false;
            foreach ( $h2s as $h ) { if ( $contains( $h, $keyword ) ) { $found_h2 = true; break; } }
            if ( $found_h2 ) $placement_score_raw += $placement_weights['h2'];
        }
        if ( $contains( $meta_desc, $keyword ) ) $placement_score_raw += $placement_weights['meta'];
        if ( $contains( $slug, $keyword ) ) $placement_score_raw += $placement_weights['slug'];
        if ( $contains( $plain, $keyword ) ) $placement_score_raw += $placement_weights['body'];

        // include link relevance bonus computed earlier
        if ( isset( $link_relevance_bonus ) && $link_relevance_bonus > 0 ) {
            $placement_score_raw += $link_relevance_bonus;
        }

        $placement_score = $placement_score_raw; // out of ~40+

        // 2) Content Quality & Structure (20%)
        $cq = 0;
        // word count scoring (8)
        if ( $words >= $min_word_count ) {
            $cq += 8;
        } elseif ( $words >= (int) ( $min_word_count / 2 ) ) {
            $cq += 4 + ( ( $words - ( $min_word_count / 2 ) ) / ( $min_word_count / 2 ) ) * 4;
        }
        // heading usage (4)
        $h2_count = count( $h2s );
        if ( $h2_count >= 2 ) { $cq += 4; } elseif ( $h2_count === 1 ) { $cq += 2; }
        // paragraph readability (3) - average paragraph length
        $paras = array_filter( array_map( 'trim', preg_split( '/<\/p>|\n\n/u', $content ) ) );
        $avg_para_words = 0;
        if ( ! empty( $paras ) ) {
            $sum = 0; $cnt = 0;
            foreach ( $paras as $p ) { $w = str_word_count( wp_strip_all_tags( $p ) ); $sum += $w; $cnt++; }
            if ( $cnt ) $avg_para_words = $sum / $cnt;
        }
        if ( $avg_para_words > 0 && $avg_para_words <= 150 ) { $cq += 3; } elseif ( $avg_para_words > 150 && $avg_para_words <= 300 ) { $cq += 1.5; }
        // internal linking (2) - useful but less influential than external references
        if ( $internal >= 2 ) { $cq += 2; } elseif ( $internal === 1 ) { $cq += 1; }
        // external linking (4) - give more weight to external authoritative references
        if ( $external >= 1 ) { $cq += 4; }

        // Structural elements that improve readability/usability: headings, lists, images, blockquotes, tables
        $heading_count = 0; preg_match_all( '/<h[1-6][^>]*>/i', $content, $mh ); if ( ! empty( $mh[0] ) ) $heading_count = count( $mh[0] );
        $list_count = 0; preg_match_all( '/<(ul|ol)[^>]*>/i', $content, $ml ); if ( ! empty( $ml[0] ) ) $list_count = count( $ml[0] );
        $list_item_count = 0; preg_match_all( '/<li[^>]*>/i', $content, $mli ); if ( ! empty( $mli[0] ) ) $list_item_count = count( $mli[0] );
        $blockquote_count = 0; preg_match_all( '/<blockquote[^>]*>/i', $content, $mbq ); if ( ! empty( $mbq[0] ) ) $blockquote_count = count( $mbq[0] );
        $table_count = 0; preg_match_all( '/<table[^>]*>/i', $content, $mt ); if ( ! empty( $mt[0] ) ) $table_count = count( $mt[0] );

        // Images were already detected earlier as $images array; count them
        $image_count = count( $images );

        // Structure bonus (up to 6 points) — rewards better document structure
        $structure_bonus = 0;
        // Headings: rewarding presence and depth
        if ( $heading_count >= 4 ) { $structure_bonus += 2; } elseif ( $heading_count >= 2 ) { $structure_bonus += 1; }
        // Lists: presence of lists or multiple list items
        if ( $list_count >= 2 || $list_item_count >= 6 ) { $structure_bonus += 2; } elseif ( $list_count >= 1 || $list_item_count >= 3 ) { $structure_bonus += 1; }
        // Images: useful for media-rich content
        if ( $image_count >= 3 ) { $structure_bonus += 2; } elseif ( $image_count >= 1 ) { $structure_bonus += 1; }
        // Blockquote/table: small bonus each
        if ( $blockquote_count >= 1 ) { $structure_bonus += 1; }
        if ( $table_count >= 1 ) { $structure_bonus += 1; }

        // Cap structure bonus
        if ( $structure_bonus > 6 ) $structure_bonus = 6;

        // Add structure bonus into content quality score
        $cq += $structure_bonus;

        $content_quality_score = $cq; // content quality + structure (approx out of 26 now)

        // 3) Semantic Coverage (15%) - secondaries proportion
        $semantic_score = 0;
        if ( empty( $secondaries ) ) {
            $semantic_score = 15; // nothing to check - full credit
        } else {
            $found = 0; foreach ( $secondaries as $s ) { if ( $contains( $plain, $s ) ) $found++; }
            $semantic_score = ( $found / count( $secondaries ) ) * 15;
        }

        // 4) Media Optimization (10%)
        $media_score = 0;
        if ( count( $images ) >= 1 ) { $media_score += 4; }
        // alt contains primary or variation
        $alt_ok = false; foreach ( $images as $img ) { if ( $contains( $img['alt'], $keyword ) ) { $alt_ok = true; break; } }
        if ( $alt_ok ) { $media_score += 4; }
        // descriptive filename
        $descriptive = 0; foreach ( $images as $img ) { if ( $img['filename'] && ! preg_match( '/^img_|^image_|^IMG_/i', $img['filename'] ) ) { $descriptive = 1; break; } }
        if ( $descriptive ) { $media_score += 2; }

        // Video detection: reward presence of actual video elements or common embed providers
        $video_count = 0;
        // <video> tags
        preg_match_all( '/<video\b[^>]*>(.*?)<\/video>/is', $content, $video_tags );
        if ( ! empty( $video_tags[0] ) ) {
            $video_count += count( $video_tags[0] );
        }
        // iframe embeds (YouTube, Vimeo, Dailymotion, Wistia etc.)
        preg_match_all( '/<iframe[^>]+src=["\']([^"\']+)["\'][^>]*>/i', $content, $iframe_matches );
        if ( ! empty( $iframe_matches[1] ) ) {
            foreach ( $iframe_matches[1] as $src ) {
                if ( preg_match( '/youtube|youtu\.be|vimeo|player\.vimeo|dailymotion|wistia|dmcdn/i', $src ) ) {
                    $video_count++;
                }
            }
        }
        // [video] shortcode
        if ( preg_match( '/\[video[\s\S]*?\]/i', $content ) ) {
            $video_count++;
        }

        if ( $video_count > 0 ) {
            // reward videos: up to 6 points toward media score
            if ( $video_count === 1 ) {
                $media_score += 4;
            } else {
                $media_score += 6;
            }
        }

        // 5) Technical SEO Basics (10%)
        $tech = 0;
        if ( ! empty( $meta_desc ) ) $tech += 3;
        if ( $has_canonical ) $tech += 3;
        if ( ! filter_var( $noindex, FILTER_VALIDATE_BOOLEAN ) ) $tech += 2;
        if ( $has_schema ) $tech += 2;

        $technical_score = $tech;

        // 6) Over-Optimization Penalty (up to -15)
        $penalty = 0;
        if ( $density > 3.5 ) {
            $penalty += min( 15, ( ( $density - 3.5 ) / 10 ) * 15 );
        }
        // headings overuse
        $heading_count = 0; $heading_with_kw = 0;
        preg_match_all( '/<h[1-6][^>]*>(.*?)<\/h[1-6]>/is', $content, $all_head );
        if ( ! empty( $all_head[1] ) ) {
            foreach ( $all_head[1] as $hh ) { $heading_count++; if ( $contains( $hh, $keyword ) ) $heading_with_kw++; }
            if ( $heading_count > 0 && $heading_with_kw / $heading_count > 0.6 ) {
                $penalty += min( 15 - $penalty, 5 );
            }
        }
        // repetition in short span: sliding window of 50 words
        if ( $words > 0 && $primary_occurrences > 3 ) {
            $tokens = preg_split( '/\s+/u', $plain_norm );
            for ( $i = 0; $i < max( 0, count( $tokens ) - 50 ); $i++ ) {
                $window = array_slice( $tokens, $i, 50 );
                $count = 0; foreach ( $window as $t ) { if ( $stem( $t ) === $stem( $normalize( $keyword ) ) ) $count++; }
                if ( $count > 5 ) { $penalty += min( 15 - $penalty, 5 ); break; }
            }
        }
        $penalty = min( 15, $penalty );

        // Final aggregation
        $raw_total = $placement_score + $content_quality_score + $semantic_score + $media_score + $technical_score;
        // Expected maximum before penalty is 100
        $final = max( 0, min( 100, $raw_total - $penalty ) );

        return [
            'keyword' => $keyword,
            'post_id' => $post_id,
            'breakdown' => [
                'placement' => round( $placement_score, 2 ),
                'content_quality' => round( $content_quality_score, 2 ),
                'semantic_coverage' => round( $semantic_score, 2 ),
                'media' => round( $media_score, 2 ),
                'technical' => round( $technical_score, 2 ),
                'penalty' => round( $penalty, 2 ),
            ],
            'final_score' => round( $final, 2 ),
            'raw_total' => round( $raw_total, 2 ),
            'words' => $words,
            'occurrences' => $primary_occurrences,
            'density' => round( $density, 3 ),
        ];
    }

    public static function rest_get_meta( $request ) {
        $post_id = (int) $request['id'];
        $title = get_post_meta( $post_id, 'vlwp_seo_meta_title', true );
        $desc  = get_post_meta( $post_id, 'vlwp_seo_meta_description', true );
        $keywords = get_post_meta( $post_id, 'vlwp_seo_keywords', true );
        $noindex = get_post_meta( $post_id, 'vlwp_seo_noindex', true );
        $nofollow = get_post_meta( $post_id, 'vlwp_seo_nofollow', true );
        $noarchive = get_post_meta( $post_id, 'vlwp_seo_noarchive', true );
        $nosnippet = get_post_meta( $post_id, 'vlwp_seo_nosnippet', true );

        return rest_ensure_response( [
            'vlwp_seo_meta_title' => $title,
            'vlwp_seo_meta_description' => $desc,
            'vlwp_seo_keywords' => $keywords,
            'vlwp_seo_noindex' => $noindex,
            'vlwp_seo_nofollow' => $nofollow,
            'vlwp_seo_noarchive' => $noarchive,
            'vlwp_seo_nosnippet' => $nosnippet,
        ] );
    }

    public static function rest_save_meta( $request ) {
        $post_id = (int) $request['id'];
        $params = $request->get_json_params();

        if ( isset( $params['vlwp_seo_meta_title'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_meta_title', sanitize_text_field( wp_unslash( $params['vlwp_seo_meta_title'] ) ) );
        }
        if ( isset( $params['vlwp_seo_meta_description'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_meta_description', sanitize_text_field( wp_unslash( $params['vlwp_seo_meta_description'] ) ) );
        }
        if ( isset( $params['vlwp_seo_keywords'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_keywords', sanitize_text_field( wp_unslash( $params['vlwp_seo_keywords'] ) ) );
        }

        if ( isset( $params['vlwp_seo_noindex'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_noindex', filter_var( $params['vlwp_seo_noindex'], FILTER_VALIDATE_BOOLEAN ) );
        }
        if ( isset( $params['vlwp_seo_nofollow'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_nofollow', filter_var( $params['vlwp_seo_nofollow'], FILTER_VALIDATE_BOOLEAN ) );
        }
        if ( isset( $params['vlwp_seo_noarchive'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_noarchive', filter_var( $params['vlwp_seo_noarchive'], FILTER_VALIDATE_BOOLEAN ) );
        }
        if ( isset( $params['vlwp_seo_nosnippet'] ) ) {
            update_post_meta( $post_id, 'vlwp_seo_nosnippet', filter_var( $params['vlwp_seo_nosnippet'], FILTER_VALIDATE_BOOLEAN ) );
        }

        return rest_ensure_response( [ 'success' => true ] );
    }

    public static function register_post_meta() {
        $args = [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'string',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ];

        register_post_meta( 'post', 'vlwp_seo_meta_title', array_merge( $args, [
            'sanitize_callback' => 'sanitize_text_field',
        ] ) );

        register_post_meta( 'post', 'vlwp_seo_meta_description', array_merge( $args, [
            'sanitize_callback' => 'sanitize_text_field',
        ] ) );

        register_post_meta( 'post', 'vlwp_seo_keywords', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ] );

        register_post_meta( 'post', 'vlwp_seo_noindex', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ] );

        register_post_meta( 'post', 'vlwp_seo_nofollow', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ] );

        register_post_meta( 'post', 'vlwp_seo_noarchive', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ] );

        register_post_meta( 'post', 'vlwp_seo_nosnippet', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ] );

        

        $page_args = $args;
        $page_args['auth_callback'] = function() {
            return current_user_can( 'edit_pages' );
        };
        register_post_meta( 'page', 'vlwp_seo_meta_title', array_merge( $page_args, [
            'sanitize_callback' => 'sanitize_text_field',
        ] ) );

        register_post_meta( 'page', 'vlwp_seo_meta_description', array_merge( $page_args, [
            'sanitize_callback' => 'sanitize_text_field',
        ] ) );

        register_post_meta( 'page', 'vlwp_seo_keywords', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'auth_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ] );

        register_post_meta( 'page', 'vlwp_seo_noindex', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ] );

        register_post_meta( 'page', 'vlwp_seo_nofollow', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ] );

        register_post_meta( 'page', 'vlwp_seo_noarchive', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ] );

        register_post_meta( 'page', 'vlwp_seo_nosnippet', [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'boolean',
            'sanitize_callback' => 'boolval',
            'auth_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ] );

        
    }

    public static function enqueue_editor_assets() {
        wp_enqueue_script(
            'vlwp-seo-post-seo',
            VLWP_SEO_URL . 'assets/js/post-seo.js?v=' . VLWP_SEO_VERSION . '.' . rand(1000, 9999),
            [ 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data', 'wp-compose' ],
            VLWP_SEO_VERSION,
            true
        );

        // Expose REST endpoint root and nonce for the editor script
        wp_localize_script( 'vlwp-seo-post-seo', 'vlwp_seo_data', [
            'rest_root' => esc_url_raw( rest_url( 'vlwp-seo/v1/' ) ),
            'nonce'     => wp_create_nonce( 'wp_rest' ),
        ] );

        // Provide translations for the editor script (JS i18n)
        if ( function_exists( 'wp_set_script_translations' ) ) {
            wp_set_script_translations( 'vlwp-seo-post-seo', 'vlwp-seo', VLWP_SEO_DIR . 'languages' );
        }

        // Also inject the JSON translations inline to avoid browser caching of the external JSON file.
        $locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
        $json_file = VLWP_SEO_DIR . 'languages/vlwp-seo-' . $locale . '.json';
        if ( file_exists( $json_file ) ) {
            $json = file_get_contents( $json_file );
            if ( $json ) {
                $inline = 'if ( window.wp && wp.i18n && wp.i18n.setLocaleData ) { wp.i18n.setLocaleData(' . $json . ', "vlwp-seo" ); }';
                wp_add_inline_script( 'vlwp-seo-post-seo', $inline, 'before' );
            }
        }
    }

    public static function output_meta_tags() {
        if ( ! is_singular( [ 'post', 'page' ] ) ) {
            return;
        }

        global $post;
        $post_id = $post->ID;

        $meta_title = get_post_meta( $post_id, 'vlwp_seo_meta_title', true );
        $meta_description = get_post_meta( $post_id, 'vlwp_seo_meta_description', true );

        // Use custom meta when present; otherwise fall back to generated values
        $title = $meta_title ? $meta_title : get_the_title( $post_id );
        if ( $meta_description ) {
            $description = $meta_description;
        } else {
            if ( has_excerpt( $post_id ) ) {
                $description = get_the_excerpt( $post_id );
            } else {
                $description = wp_trim_words( wp_strip_all_tags( $post->post_content ), 30, '...' );
            }
        }

        // Featured image
        $image_url = '';
        if ( has_post_thumbnail( $post_id ) ) {
            $image_id  = get_post_thumbnail_id( $post_id );
            $image_src = wp_get_attachment_image_src( $image_id, 'full' );
            if ( ! empty( $image_src[0] ) ) {
                $image_url = esc_url( $image_src[0] );
            }
        }

        echo '<title>' . esc_html( $title ) . '</title>' . "\n";

        if ( $description ) {
            echo '<meta name="description" content="' . esc_attr( $description ) . '">' . "\n";
        }

        echo '<meta property="og:type" content="article">' . "\n";
        echo '<meta property="og:title" content="' . esc_attr( $title ) . '">' . "\n";
        if ( $description ) {
            echo '<meta property="og:description" content="' . esc_attr( $description ) . '">' . "\n";
        }
        echo '<meta property="og:url" content="' . esc_url( get_permalink( $post_id ) ) . '">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr( get_bloginfo( 'name' ) ) . '">' . "\n";

        if ( $image_url ) {
            echo '<meta property="og:image" content="' . $image_url . '">' . "\n";
        }

        // Robots directives from post meta
        $robots = [];
        $r_noindex = get_post_meta( $post_id, 'vlwp_seo_noindex', true );
        $r_nofollow = get_post_meta( $post_id, 'vlwp_seo_nofollow', true );
        $r_noarchive = get_post_meta( $post_id, 'vlwp_seo_noarchive', true );
        $r_nosnippet = get_post_meta( $post_id, 'vlwp_seo_nosnippet', true );
        if ( filter_var( $r_noindex, FILTER_VALIDATE_BOOLEAN ) ) {
            $robots[] = 'noindex';
        }
        if ( filter_var( $r_nofollow, FILTER_VALIDATE_BOOLEAN ) ) {
            $robots[] = 'nofollow';
        }
        if ( filter_var( $r_noarchive, FILTER_VALIDATE_BOOLEAN ) ) {
            $robots[] = 'noarchive';
        }
        if ( filter_var( $r_nosnippet, FILTER_VALIDATE_BOOLEAN ) ) {
            $robots[] = 'nosnippet';
        }
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( implode( ', ', $robots ) ) . '">' . "\n";
        }
    }

    public static function maybe_buffer_for_canonical() {
        if ( ! is_singular() ) {
            return;
        }

        ob_start( function( $buffer ) {
            if ( strpos( $buffer, 'rel="canonical"' ) === false ) {
                $canonical = '<link rel="canonical" href="' . esc_url( get_permalink() ) . '">' . "\n";
                $buffer = preg_replace( '/<\/head>/i', $canonical . '</head>', $buffer, 1 );
            }
            return $buffer;
        } );
    }
}
