// VLWP SEO Sidebar - Document panel for meta title/description
(function(wp) {
    const { registerPlugin } = wp.plugins;
    const { PluginDocumentSettingPanel } = wp.editPost;
    const { TextControl, TextareaControl, ToggleControl, Spinner } = wp.components;
    const { useSelect, useDispatch } = wp.data;
    const { createElement, useState, useEffect } = wp.element;
    const { __ } = wp.i18n;

    function Sidebar() {
        // Select whole meta object and specific fields
        const meta = useSelect(s => s('core/editor').getEditedPostAttribute('meta') || {});
        const metaTitleFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_meta_title);
        const metaDescFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_meta_description);
        const metaKeywordsFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_keywords);
        const metaNoindexFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_noindex);
        const metaNofollowFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_nofollow);
        const metaNoarchiveFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_noarchive);
        const metaNosnippetFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('meta') && s('core/editor').getEditedPostAttribute('meta').vlwp_seo_nosnippet);
        const content = useSelect(s => s('core/editor').getEditedPostAttribute('content') || '');
        const postTitleFromStore = useSelect(s => s('core/editor').getEditedPostAttribute('title') || '');
        const postId = useSelect(s => s('core/editor').getCurrentPostId && s('core/editor').getCurrentPostId());
        const { editPost } = useDispatch('core/editor');

        const [metaTitle, setMetaTitle] = useState(metaTitleFromStore || '');
        const [metaDesc, setMetaDesc] = useState(metaDescFromStore || '');
        const [keywords, setKeywords] = useState(metaKeywordsFromStore || '');
        const [noindex, setNoindex] = useState( typeof metaNoindexFromStore !== 'undefined' ? metaNoindexFromStore : false );
        const [nofollow, setNofollow] = useState( typeof metaNofollowFromStore !== 'undefined' ? metaNofollowFromStore : false );
        const [noarchive, setNoarchive] = useState( typeof metaNoarchiveFromStore !== 'undefined' ? metaNoarchiveFromStore : false );
        const [nosnippet, setNosnippet] = useState( typeof metaNosnippetFromStore !== 'undefined' ? metaNosnippetFromStore : false );
        const [stats, setStats] = useState({ chars: 0, words: 0, densities: [] });
        const [scores, setScores] = useState([]); // per-keyword scores
        const [overallScore, setOverallScore] = useState(null);
        const [scoreLoading, setScoreLoading] = useState(false);

        // Debug store values
        // eslint-disable-next-line no-console
        //console.debug('vlwp-seo meta store:', { metaTitleFromStore, metaDescFromStore, metaKeywordsFromStore });

        // Helpers
        function stripHtml( html ) {
            const tmp = document.createElement('div');
            tmp.innerHTML = html || '';
            return tmp.textContent || tmp.innerText || '';
        }
        function countWords( text ) {
            if ( ! text ) return 0;
            return (text.trim().match(/\S+/g) || []).length;
        }
        function escapeRegExp( s ) {
            return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }

        // Inject CSS to hide scrollbars for the preview URL (only once)
        if ( typeof document !== 'undefined' && ! document.getElementById('vlwp-seo-inline-css') ) {
            try {
                const style = document.createElement('style');
                style.id = 'vlwp-seo-inline-css';
                style.innerHTML = '.vlwp-seo-url::-webkit-scrollbar{display:none;width:0;height:0}.vlwp-seo-url{ -webkit-overflow-scrolling: touch;}';
                document.head.appendChild(style);
            } catch (e) {
                // ignore
            }
        }

        // Sync local state when store updates (ignore undefined)
        useEffect(() => {
            if (typeof metaTitleFromStore !== 'undefined' && metaTitleFromStore !== metaTitle) {
                setMetaTitle(metaTitleFromStore || '');
            }
        }, [metaTitleFromStore]);
        useEffect(() => {
            if (typeof metaDescFromStore !== 'undefined' && metaDescFromStore !== metaDesc) {
                setMetaDesc(metaDescFromStore || '');
            }
        }, [metaDescFromStore]);
        useEffect(() => {
            if (typeof metaKeywordsFromStore !== 'undefined' && metaKeywordsFromStore !== keywords) {
                setKeywords(metaKeywordsFromStore || '');
            }
        }, [metaKeywordsFromStore]);
        useEffect(() => {
            if (typeof metaNoindexFromStore !== 'undefined' && metaNoindexFromStore !== noindex) {
                setNoindex( !! metaNoindexFromStore );
            }
        }, [metaNoindexFromStore]);
        useEffect(() => {
            if (typeof metaNofollowFromStore !== 'undefined' && metaNofollowFromStore !== nofollow) {
                setNofollow( !! metaNofollowFromStore );
            }
        }, [metaNofollowFromStore]);
        useEffect(() => {
            if (typeof metaNoarchiveFromStore !== 'undefined' && metaNoarchiveFromStore !== noarchive) {
                setNoarchive( !! metaNoarchiveFromStore );
            }
        }, [metaNoarchiveFromStore]);
        useEffect(() => {
            if (typeof metaNosnippetFromStore !== 'undefined' && metaNosnippetFromStore !== nosnippet) {
                setNosnippet( !! metaNosnippetFromStore );
            }
        }, [metaNosnippetFromStore]);

        // Compute stats when content or keywords change
        useEffect(() => {
            const text = stripHtml( content );
            const chars = text.length;
            const words = countWords( text );
            const raw = ( keywords || '' ).split(',').map( k => k.trim() ).filter( k => k.length );
            const densities = raw.map( k => {
                if ( words === 0 ) return { keyword: k, density: 0, occurrences: 0 };
                const pattern = new RegExp('\\b' + escapeRegExp( k ) + '\\b', 'giu');
                const occ = ( text.match( pattern ) || [] ).length;
                const density = ( occ / words ) * 100;
                return { keyword: k, density, occurrences: occ };
            } );
            setStats({ chars, words, densities });
        }, [content, keywords]);

        // Fetch per-keyword SEO score from REST endpoint (debounced after content/keywords changes)
        useEffect(() => {
            if ( ! postId || ! window.vlwp_seo_data ) return;

            const kwList = (keywords || '').split(',').map(k => k.trim()).filter(k => k.length);

            if ( kwList.length === 0 ) {
                setScores([]);
                setOverallScore(null);
                return;
            }

            let cancelled = false;
            setScoreLoading(true);

            // Wait at least 5 seconds after the last content/keyword change before fetching
            const timer = setTimeout( async () => {
                try {
                    const fetches = kwList.map( k => {
                        const url = window.vlwp_seo_data.rest_root + 'score/' + postId;
                            return fetch( url, {
                                method: 'POST',
                                credentials: 'same-origin',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-WP-Nonce': window.vlwp_seo_data.nonce,
                                },
                                body: JSON.stringify({ keyword: k, content: content, post_title: postTitleFromStore, meta_title: metaTitle, meta_desc: metaDesc }),
                            } ).then( async (r) => {
                                const text = await r.text();
                                let parsed = null;
                                try { parsed = text ? JSON.parse( text ) : null; } catch ( e ) { parsed = text; }
                                if ( ! r.ok ) {
                                    // include status and body for debugging
                                    console.debug( 'vlwp-seo: score fetch non-OK', { url, status: r.status, body: parsed } );
                                    return { keyword: k, error: true, status: r.status, body: parsed };
                                }
                                return { keyword: k, data: parsed };
                            } ).catch( e => ({ keyword: k, error: true, exception: String( e ) }) );
                    } );

                    const results = await Promise.all( fetches );
                    if ( cancelled ) return;

                    const good = results.filter( r => r && r.data && typeof r.data.final_score !== 'undefined' );
                    const parsed = good.map( r => ({ keyword: r.keyword, final_score: Number( r.data.final_score ), breakdown: r.data.breakdown || {} }) );
                    setScores( parsed );
                    if ( parsed.length ) {
                        const avg = parsed.reduce( (s, p) => s + p.final_score, 0 ) / parsed.length;
                        setOverallScore( Math.round( avg * 100 ) / 100 );
                    } else {
                        setOverallScore( null );
                    }
                } catch ( e ) {
                    // eslint-disable-next-line no-console
                    console.debug( 'vlwp-seo: score fetch failed', e );
                    setScores([]);
                    setOverallScore(null);
                } finally {
                    if ( ! cancelled ) setScoreLoading(false);
                }
            }, 3000 );

            return () => { cancelled = true; clearTimeout( timer ); };
        }, [ keywords, postId, content, metaTitle, metaDesc, postTitleFromStore ] );

        // Load persisted meta via REST on post change
        useEffect(() => {
            if ( ! postId || ! window.vlwp_seo_data ) return;
            const url = window.vlwp_seo_data.rest_root + 'meta/' + postId;
            fetch( url, {
                method: 'GET',
                credentials: 'same-origin',
                headers: {
                    'X-WP-Nonce': window.vlwp_seo_data.nonce,
                },
            } ).then( r => r.json() ).then( data => {
                if ( data ) {
                    if ( typeof data.vlwp_seo_meta_title !== 'undefined' ) setMetaTitle( data.vlwp_seo_meta_title || '' );
                    if ( typeof data.vlwp_seo_meta_description !== 'undefined' ) setMetaDesc( data.vlwp_seo_meta_description || '' );
                    if ( typeof data.vlwp_seo_keywords !== 'undefined' ) setKeywords( data.vlwp_seo_keywords || '' );
                    if ( typeof data.vlwp_seo_noindex !== 'undefined' ) setNoindex( !! data.vlwp_seo_noindex );
                    if ( typeof data.vlwp_seo_nofollow !== 'undefined' ) setNofollow( !! data.vlwp_seo_nofollow );
                    if ( typeof data.vlwp_seo_noarchive !== 'undefined' ) setNoarchive( !! data.vlwp_seo_noarchive );
                    if ( typeof data.vlwp_seo_nosnippet !== 'undefined' ) setNosnippet( !! data.vlwp_seo_nosnippet );
                    editPost( { meta: { ...meta, 
                      vlwp_seo_meta_title: data.vlwp_seo_meta_title, 
                      vlwp_seo_meta_description: data.vlwp_seo_meta_description, 
                      vlwp_seo_keywords: data.vlwp_seo_keywords,
                      vlwp_seo_noindex: data.vlwp_seo_noindex,
                      vlwp_seo_nofollow: data.vlwp_seo_nofollow,
                      vlwp_seo_noarchive: data.vlwp_seo_noarchive,
                      vlwp_seo_nosnippet: data.vlwp_seo_nosnippet } } );
                }
            } ).catch( e => {
                // eslint-disable-next-line no-console
                console.debug( 'vlwp-seo: fetch meta failed', e );
            } );
        }, [ postId ] );

        async function saveMetaAjax( data ) {
            if ( ! postId || ! window.vlwp_seo_data ) return;
            try {
                const url = window.vlwp_seo_data.rest_root + 'meta/' + postId;
                await fetch( url, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': window.vlwp_seo_data.nonce,
                    },
                    body: JSON.stringify( data ),
                } );
                // Also update editor store so other UI sees the change
                editPost( { meta: { ...meta, ...data } } );
            } catch ( e ) {
                // eslint-disable-next-line no-console
                console.error( 'vlwp-seo: save failed', e );
            }
        }

        function handleTitle(val) {
            setMetaTitle(val);
            saveMetaAjax( { vlwp_seo_meta_title: val } );
        }
        function handleDesc(val) {
            setMetaDesc(val);
            saveMetaAjax( { vlwp_seo_meta_description: val } );
        }
        function handleKeywords(val) {
            setKeywords(val);
            saveMetaAjax( { vlwp_seo_keywords: val } );
        }

        function handleNoindex( val ) {
            setNoindex( !! val );
            saveMetaAjax( { vlwp_seo_noindex: !! val } );
        }
        function handleNofollow( val ) {
            setNofollow( !! val );
            saveMetaAjax( { vlwp_seo_nofollow: !! val } );
        }
        function handleNoarchive( val ) {
            setNoarchive( !! val );
            saveMetaAjax( { vlwp_seo_noarchive: !! val } );
        }
        function handleNosnippet( val ) {
            setNosnippet( !! val );
            saveMetaAjax( { vlwp_seo_nosnippet: !! val } );
        }

        // Prepare current keyword array for rendering/loading
        const kwArr = (keywords || '').split(',').map(k => k.trim()).filter(k => k.length);

        // Helper: map score to color ranges
        const getScoreColor = (score) => {
            if ( typeof score !== 'number' || isNaN( score ) ) return 'inherit';
            if ( score <= 25 ) return '#d9534f'; // red
            if ( score <= 50 ) return '#f0ad4e'; // orange
            if ( score <= 65 ) return '#9ACD32'; // light green
            if ( score <= 76 ) return '#4CAF50'; // stronger green
            if ( score <= 87 ) return '#2E7D32'; // stronger/darker green
            return '#1B5E20'; // very strong green
        };

        // Compute canonical URL for preview and shorten if too long
        let canonicalUrl = ( () => {
            const current = wp.data.select('core/editor').getCurrentPost && wp.data.select('core/editor').getCurrentPost();
            if ( current && current.link ) return current.link;
            const slug = current && current.slug ? ('/' + current.slug + '/') : '';
            return ( typeof window !== 'undefined' && window.location ? (window.location.origin + slug) : slug );
        } )();
        let canonicalDisplay = canonicalUrl || '';
        const MAX_URL_LEN = 60;
        if ( canonicalDisplay.length > MAX_URL_LEN ) {
            const start = canonicalDisplay.slice(0, 40);
            const end = canonicalDisplay.slice(-17);
            canonicalDisplay = start + '…' + end;
        }

        // Prepare description display for SERP preview (truncate to 150 chars)
        const MAX_DESC_LEN = 150;
        let descDisplay = '';
        if ( nosnippet ) {
            descDisplay = '';
        } else if ( metaDesc && metaDesc.trim().length ) {
            const clean = metaDesc.trim();
            descDisplay = clean.length > MAX_DESC_LEN ? clean.slice(0, MAX_DESC_LEN) + '…' : clean;
        } else {
            const plain = stripHtml( content ).trim();
            descDisplay = plain.length > MAX_DESC_LEN ? plain.slice(0, MAX_DESC_LEN) + '…' : plain;
        }

        // Detect favicon URL (try common link rels, fall back to /favicon.ico)
        let faviconUrl = '';
        try {
            if ( typeof document !== 'undefined' ) {
                const icon = document.querySelector('link[rel~="icon"], link[rel*="shortcut"], link[rel*="apple-touch-icon"]');
                if ( icon && icon.getAttribute && icon.getAttribute('href') ) {
                    try {
                        faviconUrl = new URL( icon.getAttribute('href'), ( typeof window !== 'undefined' && window.location ? window.location.origin : '' ) ).toString();
                    } catch (e) {
                        faviconUrl = icon.getAttribute('href');
                    }
                } else if ( typeof window !== 'undefined' && window.location ) {
                    faviconUrl = window.location.origin + '/favicon.ico';
                }
            }
        } catch (e) {
            faviconUrl = '';
        }

        return createElement(
            PluginDocumentSettingPanel,
            { name: 'vlwp-seo-document-panel', title: __('SEO', 'vlwp-seo'), className: 'vlwp-seo-document-panel' },
            
            // Stats (Score, Characters, Words, Density)
            createElement( 'div', { style: { marginBottom: '0' } },
                createElement( 'div', { style: { } },
                    createElement( 'strong', null, __('Score', 'vlwp-seo') + ': ' ),
                    scoreLoading ? createElement( Spinner, { size: 20 } ) : ( overallScore !== null ? createElement( 'span', null,
                        createElement( 'strong', { style: { color: getScoreColor( overallScore ), fontWeight: 700 } }, overallScore ),
                        createElement( 'span', { style: { marginLeft: '6px', color: 'inherit' } }, '/100')
                    ) : createElement( 'span', null, __('—', 'vlwp-seo') ) )
                ),
                // per-keyword scores
                scoreLoading ? createElement( 'div', { style: { marginTop: '0', marginLeft: '6px' } },
                    kwArr.map( k => createElement( 'div', { key: k, style: { fontSize: 'inherit' } }, k + ': ', createElement( Spinner, { size: 12 } ) ) )
                ) : ( (scores || []).length ? createElement( 'div', { style: { marginTop: '0', marginLeft: '0' } },
                    scores.map( s => createElement( 'div', { key: s.keyword, style: { fontSize: 'inherit' } },
                        createElement( 'span', null, '— ' + s.keyword + ': ' ),
                        createElement( 'strong', { style: { color: getScoreColor( Number( s.final_score ) ), fontWeight: 700 } }, s.final_score ),
                        createElement( 'span', { style: { marginLeft: '4px', color: 'inherit' } }, '/100')
                    ) )
                ) : null ),

                createElement( 'div', { style: { marginTop: '10px' } } ),

                createElement( 'div', null, createElement( 'strong', null, __('Characters', 'vlwp-seo') + ': ' ), stats.chars ),
                createElement( 'div', null, createElement( 'strong', null, __('Words', 'vlwp-seo') + ': ' ), stats.words ),
                createElement( 'div', null, createElement( 'strong', null, __('Keyword Density', 'vlwp-seo') + ':' ) ),
                createElement( 'div', { style: { marginTop: '0' } } ),
                (stats.densities || []).map( d => createElement( 'div', { key: d.keyword },
                    createElement( 'span', { style: { marginLeft: '0' } }, '— ' + d.keyword + ': ' ),
                    createElement( 'span', { style: { color: (typeof d.density === 'number' && d.density > 2) ? 'red' : 'inherit', fontWeight: (typeof d.density === 'number' && d.density > 2) ? '700' : '400' } }, d.density.toFixed(2) + '%' ),
                    createElement( 'span', null, ' (' + d.occurrences + ')' )
                ) )
            ),
            
            createElement( 'div', { style: {
                paddingTop: '5px',
                paddingBottom: '5px',
                marginTop: '10px',
                marginBottom: '5px',
                borderTop: '1px solid #ccc',
             } } ),

            createElement( TextControl, {
                label: __('Keywords (comma separated)', 'vlwp-seo'),
                value: keywords,
                onChange: handleKeywords,
                style: { marginTop: '5px', marginBottom: '5px' }
            }),

            createElement( 'div', { style: {
                paddingTop: '5px',
                paddingBottom: '5px',
                marginTop: '5px',
                marginBottom: '5px',
                borderTop: '1px solid #ccc',
             } } ),

                createElement(TextControl, {
                label: __('Meta Title', 'vlwp-seo') + ' (' + (metaTitle ? metaTitle.length : 0) + ')',
                value: metaTitle,
                onChange: handleTitle
            }),
            createElement(TextareaControl, {
                label: createElement( 'span', null,
                    __('Meta Description', 'vlwp-seo'), ' ',
                    createElement( 'span', { style: { color: (metaDesc && metaDesc.length > 150) ? '#d9534f' : 'inherit', fontWeight: (metaDesc && metaDesc.length > 150) ? 700 : 'inherit' } }, '(' + (metaDesc ? metaDesc.length : 0) + ')' )
                ),
                value: metaDesc,
                onChange: handleDesc
            }),

            // Google-like SERP preview
            createElement( 'div', { style: { marginTop: '10px', padding: '8px 10px', border: '1px solid #ccc', borderRadius: '4px', background: '#fff' } },
                // Title (blue) with optional favicon to the left
                ( function() {
                    const titleText = (metaTitle && metaTitle.trim().length) ? metaTitle : ( postTitleFromStore || __('(no title)', 'vlwp-seo') );
                    const titleNode = faviconUrl ? createElement( 'span', { style: { display: 'inline-flex', alignItems: 'center' } },
                        createElement( 'img', { src: faviconUrl, alt: '', style: { width: '16px', height: '16px', marginRight: '10px', display: 'inline-block', verticalAlign: 'middle' } } ),
                        titleText
                    ) : titleText;
                    return createElement( 'div', { style: { color: '#1a0dab', fontSize: '16px', fontWeight: 600, lineHeight: '1.2' } }, titleNode );
                } )(),
                // Canonical / URL (green) - single-line with horizontal scroll if needed
                createElement( 'div', { className: 'vlwp-seo-url', style: { color: '#006621', fontSize: '13px', marginTop: '3px', whiteSpace: 'nowrap', overflowX: 'auto', maxWidth: '100%', scrollbarWidth: 'none', msOverflowStyle: 'none' } }, canonicalDisplay ),
                // Description (gray)
                createElement( 'div', { style: { color: 'inherit', fontSize: '13px', marginTop: '6px' } }, descDisplay )
            ),

            createElement( 'div', { style: { marginTop: '15px' } },
                createElement( ToggleControl, {
                    label: __('Noindex', 'vlwp-seo'),
                    checked: noindex,
                    onChange: handleNoindex
                } ),
                createElement( ToggleControl, {
                    label: __('Nofollow', 'vlwp-seo'),
                    checked: nofollow,
                    onChange: handleNofollow
                } ),
                createElement( ToggleControl, {
                    label: __('Noarchive', 'vlwp-seo'),
                    checked: noarchive,
                    onChange: handleNoarchive
                } ),
                createElement( ToggleControl, {
                    label: __('Nosnippet', 'vlwp-seo'),
                    checked: nosnippet,
                    onChange: handleNosnippet
                } )
            )
        );
    }

    registerPlugin('vlwp-seo', { render: Sidebar });
})(window.wp);
