<?php
/**
 * Uninstall handler for VLWP Backup.
 *
 * This file is executed by WordPress when the plugin is uninstalled via the
 * plugin screen. It should remove scheduled hooks, plugin options and any
 * backup files/logs created by the plugin.
 */

if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
    exit;
}

global $wpdb;

// Unschedule known plugin cron hooks.
$hooks = array(
    'vlwp_backup_run_scheduled_backup',
    'vlwp_backup_rebuild_indexes_cron',
    'vlwp_backup_check_status_cron',
    'vlwp_backup_check_external_status_cron',
);

foreach ( $hooks as $hook ) {
    if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
        wp_clear_scheduled_hook( $hook );
    } else {
        $ts = wp_next_scheduled( $hook );
        if ( $ts ) {
            wp_unschedule_event( $ts, $hook );
        }
    }
}

// Delete known plugin options (single-site and network-wide where applicable).
$options = array(
    'vlwp_backup_settings',
    'vlwp_backup_status',
    'vlwp_backup_external_status',
    'vlwp_backup_index_local',
    'vlwp_backup_index_remote',
);

foreach ( $options as $opt ) {
    delete_option( $opt );
    if ( is_multisite() ) {
        delete_site_option( $opt );
    }
}

// Remove backup files and logs produced under the uploads directory.
$upload_dir = wp_upload_dir();
$base_dir = isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : WP_CONTENT_DIR . '/uploads';
$backup_dir = trailingslashit( $base_dir ) . 'vlwp-backup/backups';
$logs_dir = WP_CONTENT_DIR . '/uploads/vlwp-backup/logs';

// Recursive remove helper.
if ( ! function_exists( 'vlwp_rrmdir' ) ) {
    function vlwp_rrmdir( $dir ) {
        if ( ! is_dir( $dir ) ) {
            return;
        }

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ( $it as $file ) {
            if ( $file->isDir() ) {
                @rmdir( $file->getRealPath() );
            } else {
                @unlink( $file->getRealPath() );
            }
        }

        @rmdir( $dir );
    }
}

vlwp_rrmdir( $backup_dir );
vlwp_rrmdir( $logs_dir );

// Attempt to remove the parent vlwp-backup folder if it's empty.
$parent = dirname( $backup_dir );
if ( is_dir( $parent ) ) {
    $items = array_diff( scandir( $parent ), array( '.', '..' ) );
    if ( empty( $items ) ) {
        @rmdir( $parent );
    }
}

// Extra cleanup: remove any remaining options that start with the plugin prefix.
if ( ! empty( $wpdb ) ) {
    $like = $wpdb->esc_like( 'vlwp_backup_' ) . '%';
    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $like ) );
    if ( is_multisite() ) {
        $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->sitemeta} WHERE meta_key LIKE %s", $like ) );
    }
}

return;
