<?php
/**
 * Backup storage helpers.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Backup_Storage {
	/**
	 * Return local backup directory.
	 *
	 * @return string
	 */
	public static function vlwp_get_local_backup_dir() {
		$upload_dir = wp_upload_dir();
		$base_dir = isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : WP_CONTENT_DIR . '/uploads';

		return trailingslashit( $base_dir ) . 'vlwp-backup/backups';
	}

	/**
	 * Get local backup file metadata list.
	 *
	 * @return array
	 */
	public static function vlwp_get_local_backups() {
		$directory = self::vlwp_get_local_backup_dir();
		if ( ! is_dir( $directory ) ) {
			return array();
		}

		$results = array();
		$files = glob( trailingslashit( $directory ) . '*.zip' );

		if ( ! is_array( $files ) ) {
			return array();
		}

		foreach ( $files as $file ) {
			$results[] = array(
				'name' => basename( $file ),
				'path' => $file,
				'size' => @filesize( $file ),
				'mtime' => @filemtime( $file ),
			);
		}

		usort(
			$results,
			static function ( $left, $right ) {
				return (int) $right['mtime'] - (int) $left['mtime'];
			}
		);

		return $results;
	}

	/**
	 * Resolve local backup path by file name.
	 *
	 * @param string $file_name File name.
	 * @return string
	 */
	public static function vlwp_get_local_backup_path_by_name( $file_name ) {
		$file_name = sanitize_file_name( (string) $file_name );
		if ( '' === $file_name ) {
			return '';
		}

		$path = trailingslashit( self::vlwp_get_local_backup_dir() ) . $file_name;
		if ( ! file_exists( $path ) || ! is_file( $path ) ) {
			return '';
		}

		return $path;
	}

	/**
	 * Delete a local backup archive.
	 *
	 * @param string $file_name File name.
	 * @return array
	 */
	public static function vlwp_delete_local_backup( $file_name ) {
		$path = self::vlwp_get_local_backup_path_by_name( $file_name );
		if ( '' === $path ) {
			return array(
				'success' => false,
				'message' => __( 'Backup file was not found.', 'vlwp-backup' ),
			);
		}

		if ( ! @unlink( $path ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup file could not be deleted.', 'vlwp-backup' ),
			);
		}

		// Remove from local index if present
		self::vlwp_delete_local_index_entry( $file_name );

		return array(
			'success' => true,
			'message' => __( 'Backup file deleted.', 'vlwp-backup' ),
		);
	}

	/**
	 * Save an uploaded backup zip into local storage.
	 *
	 * @param array $uploaded_file Uploaded file payload.
	 * @return array
	 */
	public static function vlwp_store_uploaded_backup( $uploaded_file ) {
		if ( empty( $uploaded_file['tmp_name'] ) || empty( $uploaded_file['name'] ) ) {
			return array(
				'success' => false,
				'message' => __( 'No backup file was uploaded.', 'vlwp-backup' ),
			);
		}

		$filename = sanitize_file_name( (string) $uploaded_file['name'] );
		if ( '.zip' !== strtolower( substr( $filename, -4 ) ) ) {
			return array(
				'success' => false,
				'message' => __( 'Only .zip backup files are allowed.', 'vlwp-backup' ),
			);
		}

		$target_dir = self::vlwp_get_local_backup_dir();
		if ( ! wp_mkdir_p( $target_dir ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup directory could not be created.', 'vlwp-backup' ),
			);
		}

		$target = trailingslashit( $target_dir ) . wp_unique_filename( $target_dir, $filename );
		if ( ! @move_uploaded_file( $uploaded_file['tmp_name'], $target ) ) {
			return array(
				'success' => false,
				'message' => __( 'Uploaded backup could not be stored.', 'vlwp-backup' ),
			);
		}


		// Record into local index for tracking
		VLWP_Backup_Storage::vlwp_record_local_backup(
			basename( $target ),
			array(
				'path' => $target,
				'size' => @filesize( $target ),
				'mtime' => @filemtime( $target ),
				'checksum' => file_exists( $target ) ? @hash_file( 'sha256', $target ) : '',
				'created_at' => time(),
				'type' => 'manual',
			)
		);

		return array(
			'success' => true,
			'message' => __( 'Backup uploaded successfully.', 'vlwp-backup' ),
			'file' => $target,
		);
	}

	/**
	 * Get internal index of local backups (option-backed).
	 *
	 * @return array
	 */
	public static function vlwp_get_local_index() {
		$index = get_option( 'vlwp_backup_index_local', array() );
		if ( ! is_array( $index ) ) {
			$index = array();
		}
		return $index;
	}

	/**
	 * Record a local backup entry into index.
	 *
	 * @param string $file_name File name.
	 * @param array  $meta Metadata.
	 * @return bool
	 */
	public static function vlwp_record_local_backup( $file_name, $meta = array() ) {
		$name = sanitize_file_name( (string) $file_name );
		if ( '' === $name ) {
			return false;
		}

		$path = isset( $meta['path'] ) ? (string) $meta['path'] : trailingslashit( self::vlwp_get_local_backup_dir() ) . $name;
		$entry = array(
			'name' => $name,
			'path' => $path,
			'size' => isset( $meta['size'] ) ? (int) $meta['size'] : ( file_exists( $path ) ? @filesize( $path ) : 0 ),
			'mtime' => isset( $meta['mtime'] ) ? (int) $meta['mtime'] : ( file_exists( $path ) ? @filemtime( $path ) : time() ),
			'checksum' => isset( $meta['checksum'] ) ? (string) $meta['checksum'] : ( file_exists( $path ) ? @hash_file( 'sha256', $path ) : '' ),
			'created_at' => isset( $meta['created_at'] ) ? (int) $meta['created_at'] : time(),
			// type: 'manual' or 'auto'
			'type' => isset( $meta['type'] ) ? (string) $meta['type'] : 'manual',
		);

		$index = self::vlwp_get_local_index();
		$index[ $name ] = $entry;
		update_option( 'vlwp_backup_index_local', $index, false );
		return true;
	}

	/**
	 * Remove local index entry.
	 *
	 * @param string $file_name File name.
	 * @return bool
	 */
	public static function vlwp_delete_local_index_entry( $file_name ) {
		$name = sanitize_file_name( (string) $file_name );
		if ( '' === $name ) {
			return false;
		}

		$index = self::vlwp_get_local_index();
		if ( isset( $index[ $name ] ) ) {
			unset( $index[ $name ] );
			update_option( 'vlwp_backup_index_local', $index, false );
		}
		return true;
	}

	/**
	 * Get remote index map stored in options.
	 *
	 * @return array
	 */
	public static function vlwp_get_remote_index() {
		$index = get_option( 'vlwp_backup_index_remote', array() );
		if ( ! is_array( $index ) ) {
			$index = array();
		}
		return $index;
	}

	/**
	 * Record a remote backup entry into index.
	 *
	 * @param string $remote_name Remote file name.
	 * @param array  $meta Metadata.
	 * @return bool
	 */
	public static function vlwp_record_remote_backup( $remote_name, $meta = array() ) {
		$name = sanitize_file_name( (string) $remote_name );
		if ( '' === $name ) {
			return false;
		}

		$entry = array(
			'name' => $name,
			'path' => isset( $meta['path'] ) ? (string) $meta['path'] : $name,
			'size' => isset( $meta['size'] ) ? (int) $meta['size'] : 0,
			'mtime' => isset( $meta['mtime'] ) ? (int) $meta['mtime'] : time(),
			'local_name' => isset( $meta['local_name'] ) ? (string) $meta['local_name'] : '',
			'checksum' => isset( $meta['checksum'] ) ? (string) $meta['checksum'] : '',
			'uploaded_at' => isset( $meta['uploaded_at'] ) ? (int) $meta['uploaded_at'] : time(),
			// type: 'manual' or 'auto'
			'type' => isset( $meta['type'] ) ? (string) $meta['type'] : 'manual',
		);

		$index = self::vlwp_get_remote_index();
		$index[ $name ] = $entry;
		update_option( 'vlwp_backup_index_remote', $index, false );
		return true;
	}

	/**
	 * Remove remote index entry.
	 *
	 * @param string $remote_name Remote file name.
	 * @return bool
	 */
	public static function vlwp_delete_remote_index_entry( $remote_name ) {
		$name = sanitize_file_name( (string) $remote_name );
		if ( '' === $name ) {
			return false;
		}

		$index = self::vlwp_get_remote_index();
		if ( isset( $index[ $name ] ) ) {
			unset( $index[ $name ] );
			update_option( 'vlwp_backup_index_remote', $index, false );
		}
		return true;
	}

	/**
	 * Rebuild local index by scanning local backup directory.
	 *
	 * @return array
	 */
	public static function vlwp_rebuild_local_index() {
		$files = self::vlwp_get_local_backups();
		$index = array();
		foreach ( $files as $file ) {
			$name = isset( $file['name'] ) ? (string) $file['name'] : '';
			$path = isset( $file['path'] ) ? (string) $file['path'] : '';
			$index[ $name ] = array(
				'name' => $name,
				'path' => $path,
				'size' => isset( $file['size'] ) ? (int) $file['size'] : ( file_exists( $path ) ? @filesize( $path ) : 0 ),
				'mtime' => isset( $file['mtime'] ) ? (int) $file['mtime'] : ( file_exists( $path ) ? @filemtime( $path ) : time() ),
				'checksum' => file_exists( $path ) ? @hash_file( 'sha256', $path ) : '',
				'created_at' => file_exists( $path ) ? @filemtime( $path ) : time(),
				'type' => 'manual',
			);
		}

		update_option( 'vlwp_backup_index_local', $index, false );
		return array(
			'success' => true,
			'message' => sprintf( __( 'Local index rebuilt with %d entries.', 'vlwp-backup' ), count( $index ) ),
			'count' => count( $index ),
		);
	}
}

