<?php
/**
 * Lightweight backup logger.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Backup_Logger {
	/**
	 * Write a log line if logging is enabled.
	 *
	 * @param string $level Log level.
	 * @param string $message Log message.
	 * @param array  $context Optional context.
	 */
	public static function vlwp_log( $level, $message, $context = array() ) {
		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		$enabled = ! empty( $settings['enable_logging'] );

		if ( ! $enabled ) {
			return;
		}

		$line = self::vlwp_format_line( $level, $message, $context );
		$log_path = self::vlwp_get_log_file_path( $settings );

		if ( empty( $log_path ) ) {
			error_log( $line );
			return;
		}

		$directory = dirname( $log_path );
		if ( ! file_exists( $directory ) ) {
			wp_mkdir_p( $directory );
		}

		file_put_contents( $log_path, $line . PHP_EOL, FILE_APPEND | LOCK_EX );
	}

	/**
	 * Build final line.
	 *
	 * @param string $level Log level.
	 * @param string $message Message.
	 * @param array  $context Context.
	 * @return string
	 */
	private static function vlwp_format_line( $level, $message, $context ) {
		$stamp = gmdate( 'Y-m-d H:i:s' );
		$level = strtoupper( sanitize_key( (string) $level ) );
		$message = sanitize_text_field( (string) $message );
		$context_string = '';

		if ( ! empty( $context ) ) {
			$context_string = ' ' . wp_json_encode( $context );
		}

		return '[' . $stamp . '] [' . $level . '] ' . $message . $context_string;
	}

	/**
	 * Resolve file path.
	 *
	 * @param array $settings Plugin settings.
	 * @return string
	 */
	private static function vlwp_get_log_file_path( $settings ) {
		$custom = isset( $settings['log_file_path'] ) ? (string) $settings['log_file_path'] : '';
		$custom = trim( $custom );

		if ( '' !== $custom ) {
			return $custom;
		}

		return trailingslashit( vlwp_backup_logs_dir ) . 'backup.log';
	}
}
