<?php
/**
 * Combined backups list table (local + remote).
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class VLWP_Backup_Combined_List_Table extends WP_List_Table {
	private $items_data = array();

	public function __construct( $items = array() ) {
		$this->items_data = is_array( $items ) ? $items : array();
		parent::__construct(
			array(
				'singular' => 'vlwp_backup_item',
				'plural' => 'vlwp_backup_items',
				'ajax' => false,
			)
		);
	}

	public function get_columns() {
		return array(
			'name' => __( 'Filename', 'vlwp-backup' ),
			'type' => __( 'Type', 'vlwp-backup' ),
			'source' => __( 'Source', 'vlwp-backup' ),
			'linked' => __( 'Linked', 'vlwp-backup' ),
			'size' => __( 'Size', 'vlwp-backup' ),
			'mtime' => __( 'Modified', 'vlwp-backup' ),
		);
	}

	public function prepare_items() {
		$this->items = $this->items_data;
		$this->_column_headers = array( $this->get_columns(), array(), array() );
	}

	public function column_default( $item, $column_name ) {
		$source = isset( $item['source'] ) ? $item['source'] : 'local';
		$name = isset( $item['name'] ) ? (string) $item['name'] : '';

		// preserve current filter in action links
		$current_filter = isset( $_GET['vlwp_filter'] ) ? sanitize_key( wp_unslash( $_GET['vlwp_filter'] ) ) : 'local';
		if ( ! in_array( $current_filter, array( 'local', 'external' ), true ) ) {
			$current_filter = 'local';
		}

		if ( 'name' === $column_name ) {
			if ( 'remote' === $source ) {
				$download_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_download_remote', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_download_remote_' . $name );
				$restore_url = wp_nonce_url( add_query_arg( array( 'page' => VLWP_Backup_Admin::PAGE_SLUG, 'vlwp_restore_file' => $name, 'vlwp_restore_source' => 'remote', 'vlwp_filter' => $current_filter ), admin_url( 'admin.php' ) ), 'vlwp_backup_restore_options_remote_' . $name );
				$delete_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_delete_remote', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_delete_remote_' . $name );
				$validate_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_validate_remote', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_validate_remote_' . $name );
				$actions = array(
					'download' => '<a href="' . esc_url( $download_url ) . '">' . esc_html__( 'Download', 'vlwp-backup' ) . '</a>',
					'validate' => '<a href="' . esc_url( $validate_url ) . '">' . esc_html__( 'Validate', 'vlwp-backup' ) . '</a>',
					'restore' => '<a href="' . esc_url( $restore_url ) . '">' . esc_html__( 'Restore', 'vlwp-backup' ) . '</a>',
					'delete' => '<a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . esc_js( __( 'Delete this remote backup file?', 'vlwp-backup' ) ) . '\');">' . esc_html__( 'Delete', 'vlwp-backup' ) . '</a>',
				);
				return esc_html( $name ) . $this->row_actions( $actions );
			} else {
				$download_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_download_local', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_download_local_' . $name );
				$restore_url = wp_nonce_url( add_query_arg( array( 'page' => VLWP_Backup_Admin::PAGE_SLUG, 'vlwp_restore_file' => $name, 'vlwp_restore_source' => 'local', 'vlwp_filter' => $current_filter ), admin_url( 'admin.php' ) ), 'vlwp_backup_restore_options_' . $name );
				$delete_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_delete_local', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_delete_local_' . $name );
				$validate_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_validate_local', 'file' => $name, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_validate_local_' . $name );
				$actions = array(
					'download' => '<a href="' . esc_url( $download_url ) . '">' . esc_html__( 'Download', 'vlwp-backup' ) . '</a>',
					'validate' => '<a href="' . esc_url( $validate_url ) . '">' . esc_html__( 'Validate', 'vlwp-backup' ) . '</a>',
					'restore' => '<a href="' . esc_url( $restore_url ) . '">' . esc_html__( 'Restore', 'vlwp-backup' ) . '</a>',
					'delete' => '<a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . esc_js( __( 'Delete this backup file?', 'vlwp-backup' ) ) . '\');">' . esc_html__( 'Delete', 'vlwp-backup' ) . '</a>',
				);
				return esc_html( $name ) . $this->row_actions( $actions );
			}
		}

		if ( 'source' === $column_name ) {
			return esc_html( ucfirst( $source ) );
		}

		if ( 'linked' === $column_name ) {
			if ( 'remote' === $source ) {
				$local = isset( $item['linked_to'] ) ? (string) $item['linked_to'] : '';
				if ( '' === $local ) {
					return '&mdash;';
				}
				$local_url = wp_nonce_url( add_query_arg( array( 'action' => 'vlwp_backup_download_local', 'file' => $local, 'vlwp_filter' => $current_filter ), admin_url( 'admin-post.php' ) ), 'vlwp_backup_download_local_' . $local );
				return '<a href="' . esc_url( $local_url ) . '">' . esc_html( $local ) . '</a>';
			} else {
				$links = isset( $item['linked'] ) && is_array( $item['linked'] ) ? $item['linked'] : array();
				if ( empty( $links ) ) {
					return '&mdash;';
				}
				$out = array();
				foreach ( $links as $remote ) {
					$out[] = esc_html( $remote );
				}
				return implode( '<br/>', $out );
			}
		}

		if ( 'size' === $column_name ) {
			return esc_html( size_format( (int) $item['size'] ) );
		}

		if ( 'type' === $column_name ) {
			$type = isset( $item['type'] ) ? (string) $item['type'] : '';
			if ( '' === $type ) {
				return '&mdash;';
			}
			return esc_html( ucfirst( $type ) );
		}

		if ( 'mtime' === $column_name ) {
			$mtime = isset( $item['mtime'] ) ? (int) $item['mtime'] : 0;
			if ( $mtime <= 0 ) {
				return '&mdash;';
			}

			return esc_html( gmdate( 'Y-m-d H:i:s', $mtime ) );
		}

		return '';
	}

	public function no_items() {
		esc_html_e( 'No backups available.', 'vlwp-backup' );
	}
}
