<?php
/**
 * Activation/deactivation and default setup.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Backup_Hooks {
	/**
	 * Register runtime hook helpers.
	 */
	public static function vlwp_init() {
		add_filter( 'cron_schedules', array( __CLASS__, 'vlwp_add_cron_schedules' ) );
		add_action( 'init', array( __CLASS__, 'vlwp_ensure_cron_event' ) );
		// Rebuild indexes hourly
		add_action( vlwp_backup_rebuild_indexes_cron_hook, array( __CLASS__, 'vlwp_rebuild_indexes_handler' ) );

		// Ensure the status check crons are handled by the admin instance (available even when cron runs in non-admin requests)
		add_action( 'vlwp_backup_check_status_cron', array( VLWP_Backup_Admin::vlwp_get_instance(), 'vlwp_check_last_backup_status' ) );
		add_action( 'vlwp_backup_check_external_status_cron', array( VLWP_Backup_Admin::vlwp_get_instance(), 'vlwp_check_external_status' ) );
	}

	/**
	 * Activation hook.
	 */
	public static function vlwp_activate() {
		self::vlwp_add_cron_schedules( array() );
		self::vlwp_register_default_options();
		self::vlwp_schedule_cron();
		self::vlwp_schedule_rebuild_cron();
		// schedule hourly status checks
		if ( ! wp_next_scheduled( 'vlwp_backup_check_status_cron' ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', 'vlwp_backup_check_status_cron' );
		}
		if ( ! wp_next_scheduled( 'vlwp_backup_check_external_status_cron' ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', 'vlwp_backup_check_external_status_cron' );
		}
		self::vlwp_ensure_storage_directories();
	}

	/**
	 * Deactivation hook.
	 */
	public static function vlwp_deactivate() {
		$timestamp = wp_next_scheduled( vlwp_backup_cron_hook );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, vlwp_backup_cron_hook );
		}

		$ts2 = wp_next_scheduled( vlwp_backup_rebuild_indexes_cron_hook );
		if ( $ts2 ) {
			wp_unschedule_event( $ts2, vlwp_backup_rebuild_indexes_cron_hook );
		}
	}

	/**
	 * Register default plugin options.
	 */
	private static function vlwp_register_default_options() {
		$defaults = VLWP_Backup_Core::vlwp_get_default_settings();
		$current = get_option( vlwp_backup_option_key, array() );

		if ( ! is_array( $current ) ) {
			$current = array();
		}

		update_option( vlwp_backup_option_key, wp_parse_args( $current, $defaults ), false );
	}

	/**
	 * Ensure cron event exists.
	 */
	private static function vlwp_schedule_cron() {
		if ( wp_next_scheduled( vlwp_backup_cron_hook ) ) {
			return;
		}

		wp_schedule_event( time() + MINUTE_IN_SECONDS, 'vlwp_backup_five_minutes', vlwp_backup_cron_hook );
	}

	/**
	 * Ensure cron event exists while plugin is active.
	 */
	public static function vlwp_ensure_cron_event() {
		if ( ! wp_next_scheduled( vlwp_backup_cron_hook ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'vlwp_backup_five_minutes', vlwp_backup_cron_hook );
		}

		if ( ! wp_next_scheduled( vlwp_backup_rebuild_indexes_cron_hook ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', vlwp_backup_rebuild_indexes_cron_hook );
		}

		// ensure hourly status checks are scheduled
		if ( ! wp_next_scheduled( 'vlwp_backup_check_status_cron' ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', 'vlwp_backup_check_status_cron' );
		}

		if ( ! wp_next_scheduled( 'vlwp_backup_check_external_status_cron' ) ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', 'vlwp_backup_check_external_status_cron' );
		}
	}

	/**
	 * Ensure rebuild-indexes cron exists while plugin is active.
	 */
	public static function vlwp_schedule_rebuild_cron() {
		if ( wp_next_scheduled( vlwp_backup_rebuild_indexes_cron_hook ) ) {
			return;
		}

		wp_schedule_event( time() + MINUTE_IN_SECONDS, 'hourly', vlwp_backup_rebuild_indexes_cron_hook );
	}

	/**
	 * Cron handler that rebuilds local and remote indexes.
	 */
	public static function vlwp_rebuild_indexes_handler() {
		// Rebuild local index
		if ( class_exists( 'VLWP_Backup_Storage' ) ) {
			VLWP_Backup_Storage::vlwp_rebuild_local_index();
		}

		// Rebuild remote index if storage mode is FTP/FTPS
		if ( class_exists( 'VLWP_Backup_Core' ) && class_exists( 'VLWP_Backup_Engine' ) ) {
			$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
			if ( isset( $settings['storage_mode'] ) && in_array( $settings['storage_mode'], array( 'ftp', 'ftps', 'sftp' ), true ) ) {
				VLWP_Backup_Engine::vlwp_rebuild_remote_index( $settings );
			}
		}
	}

	/**
	 * Add custom schedule slots.
	 *
	 * @param array $schedules Existing schedules.
	 * @return array
	 */
	public static function vlwp_add_cron_schedules( $schedules ) {
		if ( ! isset( $schedules['vlwp_backup_five_minutes'] ) ) {
			$schedules['vlwp_backup_five_minutes'] = array(
				'interval' => 5 * MINUTE_IN_SECONDS,
				'display' => __( 'Every 5 minutes (VLWP Backup)', 'vlwp-backup' ),
			);
		}

		return $schedules;
	}

	/**
	 * Ensure storage directories exist.
	 */
	private static function vlwp_ensure_storage_directories() {
		if ( ! wp_mkdir_p( VLWP_Backup_Storage::vlwp_get_local_backup_dir() ) ) {
			return;
		}

		wp_mkdir_p( vlwp_backup_logs_dir );
	}
}
