<?php
/**
 * Backup engine for archive create/restore.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Backup_Engine {
	/**
	 * Build backup archive.
	 *
	 * @param array $settings Plugin settings.
	 * @param array $args Runtime args.
	 * @return array
	 */
	public static function vlwp_create_backup_archive( $settings, $args ) {
		if ( ! class_exists( 'ZipArchive' ) ) {
			return array(
				'success' => false,
				'message' => __( 'ZipArchive extension is required for backups.', 'vlwp-backup' ),
			);
		}

		$timestamp = time();
		$temp_root = self::vlwp_get_temp_directory( 'backup-' . $timestamp . '-' . wp_generate_password( 6, false, false ) );
		if ( ! wp_mkdir_p( $temp_root ) ) {
			return array(
				'success' => false,
				'message' => __( 'Temporary backup directory could not be created.', 'vlwp-backup' ),
			);
		}

		$include_db = ! empty( $args['include_db'] );
		$include_wp_content = ! empty( $args['include_wp_content'] );
		$password = isset( $settings['backup_password'] ) ? (string) $settings['backup_password'] : '';
		$excluded_tables = self::vlwp_parse_lines_to_array( $settings['excluded_tables'] ?? '' );
		$excluded_paths = self::vlwp_parse_lines_to_array( $settings['excluded_paths'] ?? '' );

		$manifest = array(
			'version' => vlwp_backup_version,
			'created_at' => gmdate( 'c', $timestamp ),
			'site_url' => home_url( '/' ),
			'is_multisite' => is_multisite(),
			'include_db' => $include_db,
			'include_wp_content' => $include_wp_content,
			'origin' => isset( $args['origin'] ) ? (string) $args['origin'] : 'manual',
			'checksums' => array(),
		);

		if ( $include_db ) {
			$db_file = trailingslashit( $temp_root ) . 'database.sql';
			$db_export = self::vlwp_export_database_sql( $db_file, $excluded_tables );
			if ( ! $db_export['success'] ) {
				self::vlwp_remove_directory( $temp_root );
				return $db_export;
			}
		}

		if ( $include_wp_content ) {
			$target_content_dir = trailingslashit( $temp_root ) . 'wp-content';
			$copy = self::vlwp_copy_directory( WP_CONTENT_DIR, $target_content_dir, $excluded_paths );
			if ( ! $copy['success'] ) {
				self::vlwp_remove_directory( $temp_root );
				return $copy;
			}
		}

		$manifest['checksums'] = self::vlwp_build_manifest_checksums( $temp_root, $include_db );
		file_put_contents( trailingslashit( $temp_root ) . 'manifest.json', wp_json_encode( $manifest, JSON_PRETTY_PRINT ) );

		$backup_dir = VLWP_Backup_Storage::vlwp_get_local_backup_dir();
		if ( ! wp_mkdir_p( $backup_dir ) ) {
			self::vlwp_remove_directory( $temp_root );
			return array(
				'success' => false,
				'message' => __( 'Backup storage directory could not be created.', 'vlwp-backup' ),
			);
		}

		$file_name = self::vlwp_generate_backup_filename( $timestamp );
		$target_zip = trailingslashit( $backup_dir ) . $file_name;
		$zip_result = self::vlwp_create_zip_from_directory( $temp_root, $target_zip, $password );

		self::vlwp_remove_directory( $temp_root );

		if ( ! $zip_result['success'] ) {
			return $zip_result;
		}

		$zip_checksum = '';
		if ( file_exists( $target_zip ) ) {
			$h = @hash_file( 'sha256', $target_zip );
			if ( false !== $h ) {
				$zip_checksum = $h;
			}
		}

		return array(
			'success' => true,
			'message' => __( 'Backup created successfully.', 'vlwp-backup' ),
			'created_file' => $target_zip,
			'created_name' => basename( $target_zip ),
			'checksum' => $zip_checksum,
		);
	}

	/**
	 * Validate backup archive structure.
	 *
	 * @param string $backup_file Backup file path.
	 * @param array  $options Restore options.
	 * @return array
	 */
	public static function vlwp_validate_backup_archive( $backup_file, $options = array(), $settings = array() ) {
		if ( ! file_exists( $backup_file ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup file does not exist.', 'vlwp-backup' ),
			);
		}

		if ( '.zip' !== strtolower( substr( $backup_file, -4 ) ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup must be provided as .zip archive.', 'vlwp-backup' ),
			);
		}

		if ( ! class_exists( 'ZipArchive' ) ) {
			return array(
				'success' => false,
				'message' => __( 'ZipArchive extension is required for restore.', 'vlwp-backup' ),
			);
		}

		$defaults = array(
			'restore_db' => true,
			'restore_wp_content' => true,
		);
		$options = wp_parse_args( $options, $defaults );

		$zip = new ZipArchive();
		if ( true !== $zip->open( $backup_file ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be opened.', 'vlwp-backup' ),
			);
		}

		$password = isset( $settings['backup_password'] ) ? (string) $settings['backup_password'] : '';
		if ( '' !== $password ) {
			$zip->setPassword( $password );
		}

		$has_manifest = false;
		$has_db = false;
		$has_wp_content = false;
		$requires_password = false;

		for ( $index = 0; $index < $zip->numFiles; $index++ ) {
			$entry = $zip->getNameIndex( $index );
			if ( false === $entry || self::vlwp_has_unsafe_zip_path( $entry ) ) {
				$zip->close();
				return array(
					'success' => false,
					'message' => __( 'Backup archive contains unsafe file paths.', 'vlwp-backup' ),
				);
			}

			if ( 'manifest.json' === $entry ) {
				$has_manifest = true;
			}
			if ( 'database.sql' === $entry ) {
				$has_db = true;
			}
			if ( 0 === strpos( $entry, 'wp-content/' ) ) {
				$has_wp_content = true;
			}
		}

		$requires_password = self::vlwp_zip_requires_password( $zip );
		if ( ! self::vlwp_can_read_zip_entry( $zip, 'manifest.json' ) ) {
			$zip->close();

			if ( $requires_password ) {
				return array(
					'success' => false,
					'message' => __( 'Backup password is missing or incorrect.', 'vlwp-backup' ),
				);
			}

			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be read.', 'vlwp-backup' ),
			);
		}

		$zip->close();

		if ( ! $has_manifest ) {
			return array(
				'success' => false,
				'message' => __( 'Backup archive does not contain manifest.json.', 'vlwp-backup' ),
			);
		}

		if ( ! empty( $options['restore_db'] ) && ! $has_db ) {
			return array(
				'success' => false,
				'message' => __( 'Backup archive does not contain database.sql.', 'vlwp-backup' ),
			);
		}

		if ( ! empty( $options['restore_wp_content'] ) && ! $has_wp_content ) {
			return array(
				'success' => false,
				'message' => __( 'Backup archive does not contain wp-content data.', 'vlwp-backup' ),
			);
		}

		$report = array(
			'archive' => basename( (string) $backup_file ),
			'checks' => array(
				'manifest_present' => $has_manifest,
				'database_present' => $has_db,
				'wp_content_present' => $has_wp_content,
				'password_required' => $requires_password,
				'password_supplied' => ( '' !== $password ),
			),
		);

		return array(
			'success' => true,
			'message' => __( 'Backup archive structure is valid.', 'vlwp-backup' ),
			'report' => $report,
		);
	}

	/**
	 * Run restore validation without changing data.
	 *
	 * @param string $backup_file Backup file path.
	 * @param array  $options Restore options.
	 * @param array  $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_dry_run_validate_restore( $backup_file, $options = array(), $settings = array() ) {
		$validation = self::vlwp_validate_backup_archive( $backup_file, $options, $settings );
		if ( empty( $validation['success'] ) ) {
			return $validation;
		}

		$temp_root = self::vlwp_get_temp_directory( 'dry-run-' . time() . '-' . wp_generate_password( 6, false, false ) );
		if ( ! wp_mkdir_p( $temp_root ) ) {
			return array(
				'success' => false,
				'message' => __( 'Temporary validation directory could not be created.', 'vlwp-backup' ),
			);
		}

		$zip = new ZipArchive();
		if ( true !== $zip->open( $backup_file ) ) {
			self::vlwp_remove_directory( $temp_root );
			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be opened.', 'vlwp-backup' ),
			);
		}

		$password = isset( $settings['backup_password'] ) ? (string) $settings['backup_password'] : '';
		$requires_password = self::vlwp_zip_requires_password( $zip );
		if ( '' !== $password ) {
			$zip->setPassword( $password );
		}

		if ( ! $zip->extractTo( $temp_root ) ) {
			$zip->close();
			self::vlwp_remove_directory( $temp_root );

			if ( $requires_password ) {
				return array(
					'success' => false,
					'message' => __( 'Backup password is missing or incorrect.', 'vlwp-backup' ),
				);
			}

			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be extracted for validation.', 'vlwp-backup' ),
			);
		}
		$zip->close();

		$integrity = self::vlwp_verify_manifest_integrity( $temp_root, $options );
		self::vlwp_remove_directory( $temp_root );

		if ( empty( $integrity['success'] ) ) {
			return $integrity;
		}

		$base_report = isset( $validation['report'] ) && is_array( $validation['report'] ) ? $validation['report'] : array();
		$base_report['mode'] = 'dry-run';
		$base_report['integrity'] = isset( $integrity['report'] ) ? $integrity['report'] : array();

		return array(
			'success' => true,
			'message' => __( 'Restore dry-run validation passed.', 'vlwp-backup' ),
			'report' => $base_report,
		);
	}

	/**
	 * Restore backup archive.
	 *
	 * @param string $backup_file Backup file path.
	 * @param array  $options Restore options.
	 * @param array  $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_restore_backup_archive( $backup_file, $options, $settings ) {
		$validation = self::vlwp_validate_backup_archive( $backup_file, $options, $settings );
		if ( ! $validation['success'] ) {
			return $validation;
		}

		$temp_root = self::vlwp_get_temp_directory( 'restore-' . time() . '-' . wp_generate_password( 6, false, false ) );
		if ( ! wp_mkdir_p( $temp_root ) ) {
			return array(
				'success' => false,
				'message' => __( 'Temporary restore directory could not be created.', 'vlwp-backup' ),
			);
		}

		$zip = new ZipArchive();
		if ( true !== $zip->open( $backup_file ) ) {
			self::vlwp_remove_directory( $temp_root );
			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be opened.', 'vlwp-backup' ),
			);
		}

		$password = isset( $settings['backup_password'] ) ? (string) $settings['backup_password'] : '';
		$requires_password = self::vlwp_zip_requires_password( $zip );
		if ( '' !== $password ) {
			$zip->setPassword( $password );
		}

		if ( ! $zip->extractTo( $temp_root ) ) {
			$zip->close();
			self::vlwp_remove_directory( $temp_root );

			if ( $requires_password ) {
				return array(
					'success' => false,
					'message' => __( 'Backup password is missing or incorrect.', 'vlwp-backup' ),
				);
			}

			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be extracted.', 'vlwp-backup' ),
			);
		}
		$zip->close();

		$integrity = self::vlwp_verify_manifest_integrity( $temp_root, $options );
		if ( empty( $integrity['success'] ) ) {
			self::vlwp_remove_directory( $temp_root );
			return $integrity;
		}

		$report = isset( $validation['report'] ) && is_array( $validation['report'] ) ? $validation['report'] : array();
		$report['mode'] = 'restore';
		$report['integrity'] = isset( $integrity['report'] ) ? $integrity['report'] : array();
		$report['restored'] = array(
			'database' => ! empty( $options['restore_db'] ),
			'wp_content' => ! empty( $options['restore_wp_content'] ),
		);

		if ( ! empty( $options['restore_db'] ) ) {
			$db_path = trailingslashit( $temp_root ) . 'database.sql';
			$db_import = self::vlwp_import_database_sql( $db_path );
			if ( ! $db_import['success'] ) {
				self::vlwp_remove_directory( $temp_root );
				return $db_import;
			}
		}

		if ( ! empty( $options['restore_wp_content'] ) ) {
			$source = trailingslashit( $temp_root ) . 'wp-content';
			$copy = self::vlwp_copy_directory( $source, WP_CONTENT_DIR, array(), true );
			if ( ! $copy['success'] ) {
				self::vlwp_remove_directory( $temp_root );
				return $copy;
			}
		}

		self::vlwp_remove_directory( $temp_root );

		return array(
			'success' => true,
			'message' => __( 'Backup restored successfully.', 'vlwp-backup' ),
			'report' => $report,
		);
	}

	/**
	 * Build checksum map for manifest.
	 *
	 * @param string $temp_root Temp archive source root.
	 * @param bool   $include_db Whether DB file is included.
	 * @return array
	 */
	private static function vlwp_build_manifest_checksums( $temp_root, $include_db ) {
		$checksums = array();

		if ( $include_db ) {
			$db_file = trailingslashit( $temp_root ) . 'database.sql';
			if ( file_exists( $db_file ) ) {
				$hash = hash_file( 'sha256', $db_file );
				if ( false !== $hash ) {
					$checksums['database.sql'] = $hash;
				}
			}
		}

		return $checksums;
	}

	/**
	 * Verify extracted backup integrity against manifest.
	 *
	 * @param string $temp_root Extracted archive root.
	 * @param array  $options Restore options.
	 * @return array
	 */
	private static function vlwp_verify_manifest_integrity( $temp_root, $options ) {
		$manifest_path = trailingslashit( $temp_root ) . 'manifest.json';
		if ( ! file_exists( $manifest_path ) ) {
			return array(
				'success' => false,
				'message' => __( 'Manifest file is missing.', 'vlwp-backup' ),
			);
		}

		$manifest_raw = file_get_contents( $manifest_path );
		$manifest = is_string( $manifest_raw ) ? json_decode( $manifest_raw, true ) : null;
		if ( ! is_array( $manifest ) ) {
			return array(
				'success' => false,
				'message' => __( 'Manifest file is invalid.', 'vlwp-backup' ),
			);
		}

		if ( ! empty( $options['restore_db'] ) ) {
			$db_file = trailingslashit( $temp_root ) . 'database.sql';
			if ( ! file_exists( $db_file ) ) {
				return array(
					'success' => false,
					'message' => __( 'database.sql is missing for restore.', 'vlwp-backup' ),
				);
			}

			$expected = isset( $manifest['checksums']['database.sql'] ) ? (string) $manifest['checksums']['database.sql'] : '';
			if ( '' !== $expected ) {
				$actual = hash_file( 'sha256', $db_file );
				if ( false === $actual || ! hash_equals( $expected, $actual ) ) {
					return array(
						'success' => false,
						'message' => __( 'Backup checksum verification failed for database.sql.', 'vlwp-backup' ),
					);
				}
			}
		}

		if ( ! empty( $options['restore_wp_content'] ) ) {
			$wp_content = trailingslashit( $temp_root ) . 'wp-content';
			if ( ! is_dir( $wp_content ) ) {
				return array(
					'success' => false,
					'message' => __( 'wp-content is missing for restore.', 'vlwp-backup' ),
				);
			}
		}

		return array(
			'success' => true,
			'message' => __( 'Backup integrity verification passed.', 'vlwp-backup' ),
			'report' => array(
				'manifest_valid' => true,
				'database_checksum_verified' => ! empty( $options['restore_db'] ),
				'wp_content_present' => ! empty( $options['restore_wp_content'] ),
			),
		);
	}

	/**
	 * Check if archive appears to require password.
	 *
	 * @param ZipArchive $zip Open archive.
	 * @return bool
	 */
	private static function vlwp_zip_requires_password( $zip ) {
		for ( $index = 0; $index < $zip->numFiles; $index++ ) {
			$stat = $zip->statIndex( $index );
			if ( ! is_array( $stat ) || ! isset( $stat['encryption_method'] ) ) {
				continue;
			}

			if ( (int) $stat['encryption_method'] !== ZipArchive::EM_NONE ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Test whether one archive entry can be read.
	 *
	 * @param ZipArchive $zip Open archive.
	 * @param string     $entry Entry name.
	 * @return bool
	 */
	private static function vlwp_can_read_zip_entry( $zip, $entry ) {
		$stream = $zip->getStream( $entry );
		if ( false === $stream ) {
			return false;
		}

		fclose( $stream );

		return true;
	}

	/**
	 * Create zip archive from source directory.
	 *
	 * @param string $source_dir Source directory.
	 * @param string $target_zip Target zip file.
	 * @param string $password Optional password.
	 * @return array
	 */
	private static function vlwp_create_zip_from_directory( $source_dir, $target_zip, $password = '' ) {
		$zip = new ZipArchive();
		if ( true !== $zip->open( $target_zip, ZipArchive::CREATE | ZipArchive::OVERWRITE ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup archive could not be created.', 'vlwp-backup' ),
			);
		}

		if ( '' !== $password ) {
			$zip->setPassword( $password );
		}

		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $source_dir, FilesystemIterator::SKIP_DOTS ),
			RecursiveIteratorIterator::SELF_FIRST
		);

		$source_len = strlen( trailingslashit( $source_dir ) );
		foreach ( $iterator as $item ) {
			$path = $item->getPathname();
			$relative = substr( $path, $source_len );
			$relative = str_replace( '\\', '/', $relative );

			if ( $item->isDir() ) {
				$zip->addEmptyDir( rtrim( $relative, '/' ) );
				continue;
			}

			$zip->addFile( $path, $relative );

			if ( '' !== $password && method_exists( $zip, 'setEncryptionName' ) ) {
				$zip->setEncryptionName( $relative, ZipArchive::EM_AES_256 );
			}
		}

		$zip->close();

		return array(
			'success' => true,
			'message' => __( 'Archive created.', 'vlwp-backup' ),
		);
	}

	/**
	 * Export current database to SQL file.
	 *
	 * @param string $target_file Target SQL file.
	 * @param array  $excluded_tables Tables to exclude.
	 * @return array
	 */
	private static function vlwp_export_database_sql( $target_file, $excluded_tables = array() ) {
		global $wpdb;

		$tables = $wpdb->get_col( 'SHOW TABLES' );
		if ( empty( $tables ) || ! is_array( $tables ) ) {
			return array(
				'success' => false,
				'message' => __( 'No database tables found for export.', 'vlwp-backup' ),
			);
		}

		$exclude_map = array_fill_keys( $excluded_tables, true );
		$fp = fopen( $target_file, 'wb' );
		if ( ! $fp ) {
			return array(
				'success' => false,
				'message' => __( 'Database export file could not be created.', 'vlwp-backup' ),
			);
		}

		fwrite( $fp, "-- VLWP Backup SQL Export\n" );
		fwrite( $fp, 'SET FOREIGN_KEY_CHECKS=0;' . "\n\n" );

		foreach ( $tables as $table ) {
			$table = (string) $table;
			if ( isset( $exclude_map[ $table ] ) ) {
				continue;
			}

			$create_row = $wpdb->get_row( 'SHOW CREATE TABLE `' . esc_sql( $table ) . '`', ARRAY_N );
			$create_sql = isset( $create_row[1] ) ? $create_row[1] : '';
			if ( '' === $create_sql ) {
				continue;
			}

			fwrite( $fp, 'DROP TABLE IF EXISTS `' . $table . '`;' . "\n" );
			fwrite( $fp, $create_sql . ';' . "\n\n" );

			$rows = $wpdb->get_results( 'SELECT * FROM `' . esc_sql( $table ) . '`', ARRAY_A );
			if ( empty( $rows ) ) {
				continue;
			}

			foreach ( $rows as $row ) {
				$columns = array();
				$values = array();
				foreach ( $row as $column => $value ) {
					$columns[] = '`' . str_replace( '`', '``', (string) $column ) . '`';
					if ( null === $value ) {
						$values[] = 'NULL';
					} else {
						$values[] = "'" . esc_sql( $value ) . "'";
					}
				}

				$line = 'INSERT INTO `' . $table . '` (' . implode( ', ', $columns ) . ') VALUES (' . implode( ', ', $values ) . ');' . "\n";
				fwrite( $fp, $line );
			}

			fwrite( $fp, "\n" );
		}

		fwrite( $fp, 'SET FOREIGN_KEY_CHECKS=1;' . "\n" );
		fclose( $fp );

		return array(
			'success' => true,
			'message' => __( 'Database exported.', 'vlwp-backup' ),
		);
	}

	/**
	 * Import SQL file into current database.
	 *
	 * @param string $sql_file SQL file path.
	 * @return array
	 */
	private static function vlwp_import_database_sql( $sql_file ) {
		global $wpdb;

		if ( ! file_exists( $sql_file ) ) {
			return array(
				'success' => false,
				'message' => __( 'Restore SQL file is missing.', 'vlwp-backup' ),
			);
		}

		$handle = fopen( $sql_file, 'rb' );
		if ( ! $handle ) {
			return array(
				'success' => false,
				'message' => __( 'Restore SQL file could not be read.', 'vlwp-backup' ),
			);
		}

		$query_buffer = '';
		$in_string = false;
		$string_delimiter = '';

		while ( false !== ( $line = fgets( $handle ) ) ) {
			$trimmed = ltrim( $line );
			if ( '' === $query_buffer && ( 0 === strpos( $trimmed, '--' ) || 0 === strpos( $trimmed, '#' ) ) ) {
				continue;
			}

			$line_length = strlen( $line );
			for ( $index = 0; $index < $line_length; $index++ ) {
				$char = $line[ $index ];
				$prev = $index > 0 ? $line[ $index - 1 ] : '';

				if ( ( '"' === $char || "'" === $char ) && '\\' !== $prev ) {
					if ( ! $in_string ) {
						$in_string = true;
						$string_delimiter = $char;
					} elseif ( $string_delimiter === $char ) {
						$in_string = false;
					}
				}

				$query_buffer .= $char;

				if ( ';' === $char && ! $in_string ) {
					$query = trim( $query_buffer );
					$query_buffer = '';

					if ( '' === $query ) {
						continue;
					}

					$result = $wpdb->query( $query );
					if ( false === $result ) {
						fclose( $handle );
						return array(
							'success' => false,
							'message' => __( 'Database restore failed while running SQL query.', 'vlwp-backup' ),
						);
					}
				}
			}
		}

		fclose( $handle );

		$remaining = trim( $query_buffer );
		if ( '' !== $remaining ) {
			$result = $wpdb->query( $remaining );
			if ( false === $result ) {
				return array(
					'success' => false,
					'message' => __( 'Database restore failed while running SQL query.', 'vlwp-backup' ),
				);
			}
		}

		return array(
			'success' => true,
			'message' => __( 'Database restored.', 'vlwp-backup' ),
		);
	}

	/**
	 * Copy directory recursively.
	 *
	 * @param string $source Source directory.
	 * @param string $destination Destination directory.
	 * @param array  $excluded_paths Relative path exclusions.
	 * @param bool   $overwrite Whether destination files may be overwritten.
	 * @return array
	 */
	private static function vlwp_copy_directory( $source, $destination, $excluded_paths = array(), $overwrite = true ) {
		if ( ! is_dir( $source ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup source directory does not exist.', 'vlwp-backup' ),
			);
		}

		if ( ! wp_mkdir_p( $destination ) ) {
			return array(
				'success' => false,
				'message' => __( 'Backup destination directory could not be created.', 'vlwp-backup' ),
			);
		}

		$normalized_excludes = array_map(
			static function ( $item ) {
				$item = trim( str_replace( '\\', '/', (string) $item ), '/' );
				return $item;
			},
			$excluded_paths
		);
		$normalized_excludes[] = 'uploads/vlwp-backup';
		$normalized_excludes = array_unique( array_filter( $normalized_excludes ) );

		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $source, FilesystemIterator::SKIP_DOTS ),
			RecursiveIteratorIterator::SELF_FIRST
		);

		$source = trailingslashit( $source );
		$source_len = strlen( $source );
		foreach ( $iterator as $item ) {
			$item_path = $item->getPathname();
			$relative = str_replace( '\\', '/', substr( $item_path, $source_len ) );
			$relative = ltrim( $relative, '/' );

			if ( self::vlwp_is_excluded_path( $relative, $normalized_excludes ) ) {
				continue;
			}

			$target = trailingslashit( $destination ) . $relative;

			if ( $item->isDir() ) {
				if ( ! wp_mkdir_p( $target ) ) {
					return array(
						'success' => false,
						'message' => __( 'Could not create backup directory in archive.', 'vlwp-backup' ),
					);
				}
				continue;
			}

			$target_dir = dirname( $target );
			if ( ! is_dir( $target_dir ) && ! wp_mkdir_p( $target_dir ) ) {
				return array(
					'success' => false,
					'message' => __( 'Could not create backup file directory.', 'vlwp-backup' ),
				);
			}

			if ( file_exists( $target ) && ! $overwrite ) {
				continue;
			}

			if ( ! @copy( $item_path, $target ) ) {
				return array(
					'success' => false,
					'message' => __( 'Could not copy a file into backup set.', 'vlwp-backup' ),
				);
			}
		}

		return array(
			'success' => true,
			'message' => __( 'Directory copied.', 'vlwp-backup' ),
		);
	}

	/**
	 * Check if relative path is excluded.
	 *
	 * @param string $path Relative path.
	 * @param array  $excluded_paths Exclusions.
	 * @return bool
	 */
	private static function vlwp_is_excluded_path( $path, $excluded_paths ) {
		foreach ( $excluded_paths as $excluded ) {
			if ( '' === $excluded ) {
				continue;
			}

			if ( $path === $excluded || 0 === strpos( $path, trailingslashit( $excluded ) ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Convert multiline setting to normalized array.
	 *
	 * @param string $value Input string.
	 * @return array
	 */
	private static function vlwp_parse_lines_to_array( $value ) {
		$lines = preg_split( '/\r\n|\r|\n/', (string) $value );
		if ( ! is_array( $lines ) ) {
			return array();
		}

		$lines = array_map( 'trim', $lines );
		$lines = array_filter( $lines );

		return array_values( array_unique( $lines ) );
	}

	/**
	 * Detect unsafe zip entry.
	 *
	 * @param string $path Zip entry path.
	 * @return bool
	 */
	private static function vlwp_has_unsafe_zip_path( $path ) {
		$path = str_replace( '\\', '/', $path );

		if ( 0 === strpos( $path, '/' ) || false !== strpos( $path, '../' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Generate backup file name.
	 *
	 * @param int $timestamp Unix time.
	 * @return string
	 */
	private static function vlwp_generate_backup_filename( $timestamp ) {
		$host = wp_parse_url( home_url(), PHP_URL_HOST );
		$host = $host ? sanitize_title( $host ) : 'site';

		return 'backup-' . $host . '-' . gmdate( 'Ymd-His', $timestamp ) . '.zip';
	}

	/**
	 * Resolve temp directory.
	 *
	 * @param string $suffix Directory suffix.
	 * @return string
	 */
	private static function vlwp_get_temp_directory( $suffix ) {
		$upload_dir = wp_upload_dir();
		$base_dir = isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : WP_CONTENT_DIR . '/uploads';

		return trailingslashit( $base_dir ) . 'vlwp-backup/tmp/' . sanitize_file_name( $suffix );
	}

	/**
	 * Remove directory recursively.
	 *
	 * @param string $directory Directory path.
	 */
	private static function vlwp_remove_directory( $directory ) {
		if ( ! is_dir( $directory ) ) {
			return;
		}

		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $directory, FilesystemIterator::SKIP_DOTS ),
			RecursiveIteratorIterator::CHILD_FIRST
		);

		foreach ( $iterator as $item ) {
			if ( $item->isDir() ) {
				@rmdir( $item->getPathname() );
				continue;
			}
			@unlink( $item->getPathname() );
		}

		@rmdir( $directory );
	}

	/**
	 * Upload a local backup file to FTP/FTPS.
	 *
	 * @param string $local_file Local backup file path.
	 * @param array  $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_upload_backup_to_ftp( $local_file, $settings ) {
		if ( ! file_exists( $local_file ) ) {
			return array(
				'success' => false,
				'message' => __( 'Local backup file not found for FTP upload.', 'vlwp-backup' ),
			);
		}

		$connection_result = self::vlwp_open_ftp_connection( $settings );
		if ( empty( $connection_result['success'] ) ) {
			return $connection_result;
		}

		$remote_base = self::vlwp_get_remote_base_path( $settings );
		if ( '' !== $remote_base ) {
			$created = self::vlwp_remote_ensure_directory( $connection_result, $remote_base );
			if ( ! $created ) {
				self::vlwp_remote_close( $connection_result );
				return array(
					'success' => false,
					'message' => __( 'FTP target directory could not be prepared.', 'vlwp-backup' ),
				);
			}
		}

		$remote_file = ( '' !== $remote_base ? $remote_base . '/' : '' ) . basename( $local_file );
		$uploaded = self::vlwp_remote_put( $connection_result, $remote_file, $local_file );
		self::vlwp_remote_close( $connection_result );

		if ( ! $uploaded ) {
			return array(
				'success' => false,
				'message' => __( 'Backup file could not be uploaded to FTP.', 'vlwp-backup' ),
			);
		}

		return array(
			'success' => true,
			'message' => __( 'Backup uploaded to FTP storage.', 'vlwp-backup' ),
			'remote_file' => $remote_file,
		);
	}

	/**
	 * List remote backups from FTP/FTPS storage.
	 *
	 * @param array $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_list_remote_backups( $settings ) {
		// Always fetch live from remote storage (no transient cache) to keep listings real-time

		$connection_result = self::vlwp_open_ftp_connection( $settings );
		if ( empty( $connection_result['success'] ) ) {
			return array();
		}

		$remote_base = self::vlwp_get_remote_base_path( $settings );
		$remote_dir = '' !== $remote_base ? $remote_base : '.';

		$list = self::vlwp_remote_nlist( $connection_result, $remote_dir );
		if ( ! is_array( $list ) || empty( $list ) ) {
			self::vlwp_remote_close( $connection_result );
			return array();
		}

		$items = array();
		foreach ( $list as $entry ) {
			$entry = str_replace( '\\', '/', (string) $entry );
			$name = basename( $entry );
			if ( '.zip' !== strtolower( substr( $name, -4 ) ) ) {
				continue;
			}

			$path = '' !== $remote_base ? $remote_base . '/' . $name : $name;
			$items[] = array(
				'name' => $name,
				'path' => $path,
				'size' => self::vlwp_remote_filesize( $connection_result, $path ),
				'mtime' => self::vlwp_remote_mtime( $connection_result, $path ),
			);
		}

		self::vlwp_remote_close( $connection_result );

		usort(
			$items,
			static function ( $left, $right ) {
				return (int) $right['mtime'] - (int) $left['mtime'];
			}
		);

		return $items;
	}

	/**
	 * Rebuild remote index stored in options by scanning remote storage.
	 *
	 * @param array $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_rebuild_remote_index( $settings ) {
		// No transient cache to clear — listings are fetched live

		$list = self::vlwp_list_remote_backups( $settings );
		if ( empty( $list ) ) {
			// ensure remote index is cleared
			update_option( 'vlwp_backup_index_remote', array(), false );
			return array(
				'success' => true,
				'message' => __( 'Remote index rebuilt (no remote files found).', 'vlwp-backup' ),
				'count' => 0,
			);
		}

		// Clear existing index
		update_option( 'vlwp_backup_index_remote', array(), false );

		$local_index = VLWP_Backup_Storage::vlwp_get_local_index();
		$added = 0;
		foreach ( $list as $item ) {
			$name = isset( $item['name'] ) ? (string) $item['name'] : '';
			$meta = array(
				'path' => isset( $item['path'] ) ? (string) $item['path'] : $name,
				'size' => isset( $item['size'] ) ? (int) $item['size'] : 0,
				'mtime' => isset( $item['mtime'] ) ? (int) $item['mtime'] : time(),
				'local_name' => isset( $local_index[ $name ] ) ? $name : '',
				'checksum' => '',
				'uploaded_at' => isset( $item['mtime'] ) ? (int) $item['mtime'] : time(),
			);

			VLWP_Backup_Storage::vlwp_record_remote_backup( $name, $meta );
			$added++;
		}

		return array(
			'success' => true,
			'message' => sprintf( __( 'Remote index rebuilt with %d entries.', 'vlwp-backup' ), $added ),
			'count' => $added,
		);
	}

	/**
	 * Download one remote backup file to a local path.
	 *
	 * @param string $remote_name Remote file name.
	 * @param string $local_path Local file path.
	 * @param array  $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_download_remote_backup( $remote_name, $local_path, $settings ) {
		$connection_result = self::vlwp_open_ftp_connection( $settings );
		if ( empty( $connection_result['success'] ) ) {
			return $connection_result;
		}

		$remote_name = sanitize_file_name( (string) $remote_name );
		$remote_file = self::vlwp_build_remote_file_path( $settings, $remote_name );

		$local_dir = dirname( $local_path );
		if ( ! is_dir( $local_dir ) && ! wp_mkdir_p( $local_dir ) ) {
			self::vlwp_remote_close( $connection_result );
			return array(
				'success' => false,
				'message' => __( 'Local download directory could not be created.', 'vlwp-backup' ),
			);
		}

		$downloaded = self::vlwp_remote_get( $connection_result, $remote_file, $local_path );
		self::vlwp_remote_close( $connection_result );

		if ( ! $downloaded ) {
			return array(
				'success' => false,
				'message' => __( 'Remote backup could not be downloaded.', 'vlwp-backup' ),
			);
		}

		return array(
			'success' => true,
			'message' => __( 'Remote backup downloaded.', 'vlwp-backup' ),
			'file' => $local_path,
		);
	}

	/**
	 * Delete one remote backup file.
	 *
	 * @param string $remote_name Remote file name.
	 * @param array  $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_delete_remote_backup( $remote_name, $settings ) {
		$connection_result = self::vlwp_open_ftp_connection( $settings );
		if ( empty( $connection_result['success'] ) ) {
			return $connection_result;
		}

		$remote_name = sanitize_file_name( (string) $remote_name );
		$remote_file = self::vlwp_build_remote_file_path( $settings, $remote_name );

		$deleted = self::vlwp_remote_delete( $connection_result, $remote_file );
		self::vlwp_remote_close( $connection_result );

		if ( ! $deleted ) {
			return array(
				'success' => false,
				'message' => __( 'Remote backup could not be deleted.', 'vlwp-backup' ),
			);
		}

		return array(
			'success' => true,
			'message' => __( 'Remote backup deleted.', 'vlwp-backup' ),
		);
	}

	/**
	 * Test FTP/FTPS connectivity with current settings.
	 *
	 * @param array $settings Plugin settings.
	 * @return array
	 */
	public static function vlwp_test_ftp_connection( $settings ) {
		$connection_result = self::vlwp_open_ftp_connection( $settings );
		if ( empty( $connection_result['success'] ) ) {
			return $connection_result;
		}

		$remote_base = self::vlwp_get_remote_base_path( $settings );

		if ( '' !== $remote_base ) {
			$created = self::vlwp_remote_ensure_directory( $connection_result, $remote_base );
			if ( ! $created ) {
				self::vlwp_remote_close( $connection_result );
				return array(
					'success' => false,
					'message' => __( 'Connected, but FTP path is not writable or cannot be created.', 'vlwp-backup' ),
				);
			}
		}

		self::vlwp_remote_close( $connection_result );

		return array(
			'success' => true,
			'message' => __( 'External storage connection successful.', 'vlwp-backup' ),
		);
	}


	/**
	 * Open FTP/FTPS connection from settings.
	 *
	 * @param array $settings Plugin settings.
	 * @return array
	 */
	private static function vlwp_open_ftp_connection( $settings ) {
		$host = isset( $settings['ftp_host'] ) ? trim( (string) $settings['ftp_host'] ) : '';
		$user = isset( $settings['ftp_username'] ) ? (string) $settings['ftp_username'] : '';
		$pass = isset( $settings['ftp_password'] ) ? (string) $settings['ftp_password'] : '';
		$port = max( 1, (int) ( $settings['ftp_port'] ?? 21 ) );

		if ( '' === $host || '' === $user || '' === $pass ) {
			return array(
				'success' => false,
				'message' => __( 'FTP settings are incomplete.', 'vlwp-backup' ),
			);
		}

		$mode = isset( $settings['storage_mode'] ) ? (string) $settings['storage_mode'] : 'ftp';

		// SFTP via phpseclib
		if ( 'sftp' === $mode ) {
			if ( ! class_exists( '\phpseclib3\Net\SFTP' ) ) {
				return array(
					'success' => false,
					'message' => __( 'phpseclib SFTP library not available.', 'vlwp-backup' ),
				);
			}

			try {
				$sftp = new \phpseclib3\Net\SFTP( $host, $port );
				if ( ! $sftp->login( $user, $pass ) ) {
					return array( 'success' => false, 'message' => __( 'SFTP login failed.', 'vlwp-backup' ) );
				}

				return array( 'success' => true, 'type' => 'sftp', 'connection' => $sftp );
			} catch ( Exception $e ) {
				return array( 'success' => false, 'message' => __( 'SFTP connection error.', 'vlwp-backup' ) );
			}
		}

		// FTP/FTPS fallback
		$use_ssl = ! empty( $settings['ftp_ssl'] ) || 'ftps' === $mode;

		if ( ! function_exists( 'ftp_connect' ) ) {
			return array(
				'success' => false,
				'message' => __( 'FTP extension is not available in PHP.', 'vlwp-backup' ),
			);
		}

		$connection = false;
		if ( $use_ssl && function_exists( 'ftp_ssl_connect' ) ) {
			$connection = @ftp_ssl_connect( $host, $port, 20 );
		}
		if ( ! $connection ) {
			$connection = @ftp_connect( $host, $port, 20 );
		}

		if ( ! $connection ) {
			return array(
				'success' => false,
				'message' => __( 'FTP connection could not be established.', 'vlwp-backup' ),
			);
		}

		if ( ! @ftp_login( $connection, $user, $pass ) ) {
			@ftp_close( $connection );
			return array(
				'success' => false,
				'message' => __( 'FTP login failed.', 'vlwp-backup' ),
			);
		}

		@ftp_pasv( $connection, true );

		return array( 'success' => true, 'type' => 'ftp', 'connection' => $connection );
	}

	/**
	 * Close remote connection (FTP resource or phpseclib SFTP object).
	 */
	private static function vlwp_remote_close( $conn_info ) {
		if ( ! is_array( $conn_info ) || empty( $conn_info['connection'] ) ) {
			return;
		}
		$connection = $conn_info['connection'];
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		if ( 'sftp' === $type && is_object( $connection ) && method_exists( $connection, 'disconnect' ) ) {
			$connection->disconnect();
			return;
		}
		if ( is_resource( $connection ) ) {
			@ftp_close( $connection );
		}
	}

	/**
	 * Ensure remote directory exists for FTP or SFTP.
	 */
	private static function vlwp_remote_ensure_directory( $conn_info, $directory ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		$parts = array_filter( explode( '/', $directory ) );
		$current = '';
		foreach ( $parts as $part ) {
			$current .= '/' . $part;
			if ( 'sftp' === $type && is_object( $connection ) ) {
				// try chdir via stat
				if ( $connection->is_dir( $current ) ) {
					continue;
				}
				if ( ! $connection->mkdir( $current ) ) {
					return false;
				}
			} else {
				if ( @ftp_chdir( $connection, $current ) ) {
					@ftp_chdir( $connection, '/' );
					continue;
				}
				if ( false === @ftp_mkdir( $connection, $current ) ) {
					return false;
				}
			}
		}
		if ( 'sftp' !== $type && is_resource( $connection ) ) {
			@ftp_chdir( $connection, '/' );
		}
		return true;
	}

	/**
	 * Remote nlist wrapper.
	 */
	private static function vlwp_remote_nlist( $conn_info, $remote_dir ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			$list = $connection->nlist( $remote_dir );
			if ( ! is_array( $list ) ) {
				return array();
			}
			return $list;
		}
		$list = @ftp_nlist( $connection, $remote_dir );
		return is_array( $list ) ? $list : array();
	}

	/**
	 * Remote filesize wrapper.
	 */
	private static function vlwp_remote_filesize( $conn_info, $path ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			$stat = $connection->stat( $path );
			if ( is_array( $stat ) && isset( $stat['size'] ) ) {
				return (int) $stat['size'];
			}
			return 0;
		}
		return max( 0, (int) @ftp_size( $connection, $path ) );
	}

	/**
	 * Remote mtime wrapper.
	 */
	private static function vlwp_remote_mtime( $conn_info, $path ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			$stat = $connection->stat( $path );
			if ( is_array( $stat ) && isset( $stat['mtime'] ) ) {
				return (int) $stat['mtime'];
			}
			return 0;
		}
		return max( 0, (int) @ftp_mdtm( $connection, $path ) );
	}

	/**
	 * Remote put wrapper (upload local file).
	 */
	private static function vlwp_remote_put( $conn_info, $remote_file, $local_file ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			return (bool) $connection->put( $remote_file, $local_file, \phpseclib3\Net\SFTP::SOURCE_LOCAL_FILE );
		}
		return @ftp_put( $connection, $remote_file, $local_file, FTP_BINARY );
	}

	/**
	 * Remote get wrapper (download remote to local).
	 */
	private static function vlwp_remote_get( $conn_info, $remote_file, $local_file ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			return (bool) $connection->get( $remote_file, $local_file );
		}
		return @ftp_get( $connection, $local_file, $remote_file, FTP_BINARY );
	}

	/**
	 * Remote delete wrapper.
	 */
	private static function vlwp_remote_delete( $conn_info, $remote_file ) {
		$type = isset( $conn_info['type'] ) ? $conn_info['type'] : 'ftp';
		$connection = $conn_info['connection'];
		if ( 'sftp' === $type && is_object( $connection ) ) {
			return (bool) $connection->delete( $remote_file );
		}
		return @ftp_delete( $connection, $remote_file );
	}

	/**
	 * Build base remote directory path.
	 *
	 * @param array $settings Plugin settings.
	 * @return string
	 */
	private static function vlwp_get_remote_base_path( $settings ) {
		$remote_base = isset( $settings['ftp_path'] ) ? trim( (string) $settings['ftp_path'] ) : '';
		return trim( str_replace( '\\', '/', $remote_base ), '/' );
	}

	/**
	 * Build one remote file path from base and name.
	 *
	 * @param array  $settings Plugin settings.
	 * @param string $file_name File name.
	 * @return string
	 */
	private static function vlwp_build_remote_file_path( $settings, $file_name ) {
		$base = self::vlwp_get_remote_base_path( $settings );
		return '' !== $base ? $base . '/' . $file_name : $file_name;
	}
}
