<?php
/**
 * Core backup orchestration.
 */

defined( 'ABSPATH' ) || exit;

class VLWP_Backup_Core {
	/**
	 * Singleton instance.
	 *
	 * @var VLWP_Backup_Core|null
	 */
	private static $instance = null;

	/**
	 * Last scheduled run option key.
	 *
	 * @var string
	 */
	const OPTION_LAST_SCHEDULED_RUN = 'vlwp_backup_last_scheduled_run';

	/**
	 * Backup lock option key.
	 *
	 * @var string
	 */
	const OPTION_BACKUP_LOCK = 'vlwp_backup_lock_backup';

	/**
	 * Restore lock option key.
	 *
	 * @var string
	 */
	const OPTION_RESTORE_LOCK = 'vlwp_backup_lock_restore';

	/**
	 * Lock timeout in seconds.
	 *
	 * @var int
	 */
	const LOCK_TIMEOUT = 1800;

	/**
	 * Get singleton instance.
	 *
	 * @return VLWP_Backup_Core
	 */
	public static function vlwp_get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Send a dedicated notification when external storage check fails.
	 *
	 * @param array $result Connection test result payload.
	 * @param array $settings Plugin settings.
	 */
	private function vlwp_send_external_storage_error_notification( $result, $settings ) {
		if ( empty( $settings['email_notification_enabled'] ) ) {
			return;
		}

		$to = isset( $settings['notification_email'] ) ? sanitize_email( $settings['notification_email'] ) : '';
		if ( '' === $to ) {
			return;
		}

		$subject = __( 'There is an error with the external backup storage', 'vlwp-backup' );

		$message_lines = array();
		$message_lines[] = __( 'External backup storage check failed.', 'vlwp-backup' );
		$message_lines[] = '';
		if ( isset( $result['message'] ) && '' !== (string) $result['message'] ) {
			$message_lines[] = __( 'Error message:', 'vlwp-backup' );
			$message_lines[] = (string) $result['message'];
			$message_lines[] = '';
		}
		// Additional details
		$mode = isset( $settings['storage_mode'] ) ? (string) $settings['storage_mode'] : 'local';
		$message_lines[] = __( 'Storage mode:', 'vlwp-backup' ) . ' ' . $mode;
		if ( ! empty( $settings['ftp_host'] ) ) {
			$message_lines[] = __( 'Host:', 'vlwp-backup' ) . ' ' . sanitize_text_field( $settings['ftp_host'] );
		}
		$message_lines[] = __( 'Checked at:', 'vlwp-backup' ) . ' ' . date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), time() );

		$message = implode( "\n", $message_lines );

		wp_mail( $to, $subject, $message );
	}

	/**
	 * Register runtime hooks.
	 */
	public function vlwp_init() {
		add_action( vlwp_backup_cron_hook, array( $this, 'vlwp_run_scheduled_backup' ) );
	}

	/**
	 * Return merged settings.
	 *
	 * @return array
	 */
	public function vlwp_get_settings() {
		$saved = get_option( vlwp_backup_option_key, array() );
		if ( ! is_array( $saved ) ) {
			$saved = array();
		}

		return wp_parse_args( $saved, self::vlwp_get_default_settings() );
	}

	/**
	 * Default plugin settings.
	 *
	 * @return array
	 */
	public static function vlwp_get_default_settings() {
		return array(
			'schedule_enabled' => 0,
			'schedule_interval_value' => 12,
			'schedule_interval_unit' => 'hours',
			'storage_mode' => 'local',
			'ftp_host' => '',
			'ftp_port' => 22,
			'ftp_username' => '',
			'ftp_password' => '',
			'ftp_path' => '',
			'ftp_ssl' => 0,
			'backup_password' => '',
			'email_notification_enabled' => 0,
			'email_attach_backup' => 0,
			'notification_email' => get_bloginfo( 'admin_email' ),
			'excluded_paths' => '',
			'excluded_tables' => '',
			'retention_enabled' => 1,
			'retention_days' => 30,
			'retention_max_backups' => 20,
			// Remote retention defaults (apply to external copies)
			'remote_retention_enabled' => 1,
			'remote_retention_days' => 30,
			'remote_retention_max_backups' => 20,
			'enable_logging' => 1,
			'log_file_path' => '',
			'multisite_scope' => 'site',
		);
	}

	/**
	 * Trigger backup from scheduler.
	 */
	public function vlwp_run_scheduled_backup() {
		$settings = $this->vlwp_get_settings();
		if ( empty( $settings['schedule_enabled'] ) ) {
			return;
		}

		if ( ! $this->vlwp_is_schedule_due( $settings ) ) {
			return;
		}

		$result = $this->vlwp_run_backup(
			array(
				'origin' => 'scheduler',
				'include_db' => true,
				'include_wp_content' => true,
			)
		);

		if ( ! empty( $result['success'] ) ) {
			update_option( self::OPTION_LAST_SCHEDULED_RUN, time(), false );
		}
	}

	/**
	 * Run backup process.
	 *
	 * @param array $args Runtime arguments.
	 * @return array
	 */
	public function vlwp_run_backup( $args = array() ) {
		$lock_context = isset( $args['origin'] ) ? (string) $args['origin'] : 'manual';
		if ( ! $this->vlwp_acquire_lock( self::OPTION_BACKUP_LOCK, $lock_context ) ) {
			return array(
				'success' => false,
				'message' => __( 'Another backup process is already running.', 'vlwp-backup' ),
			);
		}

		try {
		$defaults = array(
			'origin' => 'manual',
			'include_db' => true,
			'include_wp_content' => true,
			'is_test' => false,
		);
		$args = wp_parse_args( $args, $defaults );

		$settings = $this->vlwp_get_settings();
		$result = array();

		if ( ! empty( $settings['email_attach_backup'] ) && empty( $settings['backup_password'] ) ) {
			$result = array(
				'success' => false,
				'message' => __( 'Backup email attachment requires backup password protection.', 'vlwp-backup' ),
				'created_file' => '',
				'context' => $args,
			);
		} else {
			$result = VLWP_Backup_Engine::vlwp_create_backup_archive( $settings, $args );
			$result['context'] = $args;

			// Record local backup metadata into index for tracking.
			if ( ! empty( $result['success'] ) && ! empty( $result['created_file'] ) ) {
				$created_file = (string) $result['created_file'];
				$created_name = basename( $created_file );
				$checksum = isset( $result['checksum'] ) ? (string) $result['checksum'] : ( file_exists( $created_file ) ? @hash_file( 'sha256', $created_file ) : '' );
				VLWP_Backup_Storage::vlwp_record_local_backup(
					$created_name,
					array(
						'path' => $created_file,
						'size' => @filesize( $created_file ),
						'mtime' => @filemtime( $created_file ),
						'checksum' => $checksum,
						'created_at' => time(),
						'type' => ( isset( $args['origin'] ) && 'scheduler' === $args['origin'] ) ? 'auto' : 'manual',
					)
				);
			}

			if ( ! empty( $result['success'] ) ) {
				if ( in_array( $settings['storage_mode'], array( 'ftp', 'ftps', 'sftp' ), true ) ) {
					$upload = VLWP_Backup_Engine::vlwp_upload_backup_to_ftp( (string) $result['created_file'], $settings );
					if ( empty( $upload['success'] ) ) {
						$result['success'] = false;
						$result['message'] = isset( $upload['message'] ) ? (string) $upload['message'] : __( 'FTP upload failed.', 'vlwp-backup' );
					} else {
						$result['remote_file'] = isset( $upload['remote_file'] ) ? (string) $upload['remote_file'] : '';
						$result['message'] = __( 'Backup created and uploaded to external storage.', 'vlwp-backup' );

						// Record remote copy mapping to local entry.
						$remote_name = isset( $upload['remote_file'] ) ? basename( (string) $upload['remote_file'] ) : '';
						if ( '' !== $remote_name ) {
							VLWP_Backup_Storage::vlwp_record_remote_backup(
								$remote_name,
								array(
									'path' => (string) $upload['remote_file'],
									'size' => file_exists( $result['created_file'] ) ? @filesize( $result['created_file'] ) : 0,
									'mtime' => time(),
									'local_name' => isset( $created_name ) ? $created_name : '',
									'checksum' => isset( $checksum ) ? $checksum : '',
									'uploaded_at' => time(),
									'type' => ( isset( $args['origin'] ) && 'scheduler' === $args['origin'] ) ? 'auto' : 'manual',
								)
							);
						}
					}
				}

				$this->vlwp_apply_retention_policy();
			}
		}

		$this->vlwp_maybe_send_backup_notification( $result, $settings );

		VLWP_Backup_Logger::vlwp_log(
			$result['success'] ? 'info' : 'error',
			'Backup run finished.',
			array(
				'origin' => $args['origin'],
				'success' => $result['success'],
			)
		);

		return $result;
		} finally {
			$this->vlwp_release_lock( self::OPTION_BACKUP_LOCK );
		}
	}

	/**
	 * Validate backup package before restore.
	 *
	 * @param string $backup_file Backup archive path.
	 * @return array
	 */
	public function vlwp_validate_backup_before_restore( $backup_file ) {
		$settings = $this->vlwp_get_settings();
		return VLWP_Backup_Engine::vlwp_validate_backup_archive( $backup_file, array(), $settings );
	}

	/**
	 * Validate backup restore path without executing restore.
	 *
	 * @param string $backup_file Backup archive path.
	 * @param array  $options Validation options.
	 * @return array
	 */
	public function vlwp_validate_restore_dry_run( $backup_file, $options = array() ) {
		$defaults = array(
			'restore_db' => true,
			'restore_wp_content' => true,
		);
		$options = wp_parse_args( $options, $defaults );
		$settings = $this->vlwp_get_settings();

		$result = VLWP_Backup_Engine::vlwp_dry_run_validate_restore( $backup_file, $options, $settings );

		VLWP_Backup_Logger::vlwp_log(
			empty( $result['success'] ) ? 'error' : 'info',
			'Restore dry-run validation finished.',
			array(
				'file' => basename( (string) $backup_file ),
				'success' => ! empty( $result['success'] ),
			)
		);

		return $result;
	}

	/**
	 * Run restore flow.
	 *
	 * @param string $backup_file Backup archive path.
	 * @param array  $options Restore options.
	 * @return array
	 */
	public function vlwp_restore_backup( $backup_file, $options = array() ) {
		if ( ! $this->vlwp_acquire_lock( self::OPTION_RESTORE_LOCK, basename( (string) $backup_file ) ) ) {
			return array(
				'success' => false,
				'message' => __( 'Another restore process is already running.', 'vlwp-backup' ),
			);
		}

		try {
		$defaults = array(
			'restore_db' => true,
			'restore_wp_content' => true,
		);
		$options = wp_parse_args( $options, $defaults );

		$validation = VLWP_Backup_Engine::vlwp_validate_backup_archive( $backup_file, $options, $this->vlwp_get_settings() );
		if ( empty( $validation['success'] ) ) {
			return $validation;
		}

		$settings = $this->vlwp_get_settings();
		$result = VLWP_Backup_Engine::vlwp_restore_backup_archive( $backup_file, $options, $settings );

		VLWP_Backup_Logger::vlwp_log(
			$result['success'] ? 'info' : 'error',
			'Restore run finished.',
			array(
				'file' => basename( $backup_file ),
				'restore_db' => ! empty( $options['restore_db'] ),
				'restore_wp_content' => ! empty( $options['restore_wp_content'] ),
				'success' => ! empty( $result['success'] ),
			)
		);

		return $result;
		} finally {
			$this->vlwp_release_lock( self::OPTION_RESTORE_LOCK );
		}
	}

	/**
	 * Run test backup and restore sequence.
	 *
	 * @return array
	 */
	public function vlwp_run_backup_restore_test() {
		$backup = $this->vlwp_run_backup(
			array(
				'origin' => 'test',
				'is_test' => true,
			)
		);

		if ( empty( $backup['success'] ) ) {
			return $backup;
		}

		$created = isset( $backup['created_file'] ) ? (string) $backup['created_file'] : '';
		if ( '' === $created ) {
			return array(
				'success' => false,
				'message' => __( 'Test backup file was not created.', 'vlwp-backup' ),
			);
		}

		return $this->vlwp_restore_backup(
			$created,
			array(
				'restore_db' => true,
				'restore_wp_content' => false,
			)
		);
	}

	/**
	 * Test configured external storage connection.
	 *
	 * @return array
	 */
	public function vlwp_test_remote_connection() {
		$settings = $this->vlwp_get_settings();

		if ( ! in_array( $settings['storage_mode'], array( 'ftp', 'ftps', 'sftp' ), true ) ) {
			return array(
				'success' => false,
				'message' => __( 'External storage is not enabled. Switch storage mode to FTP, FTPS or SFTP first.', 'vlwp-backup' ),
			);
		}

		$result = VLWP_Backup_Engine::vlwp_test_ftp_connection( $settings );

		VLWP_Backup_Logger::vlwp_log(
			empty( $result['success'] ) ? 'error' : 'info',
			'External connection test finished.',
			array(
				'success' => ! empty( $result['success'] ),
			)
		);

		// If external connection failed and notifications are enabled, send a dedicated notification email.
		if ( empty( $result['success'] ) && ! empty( $settings['email_notification_enabled'] ) ) {
			$this->vlwp_send_external_storage_error_notification( $result, $settings );
		}

		return $result;
	}

	/**
	 * Check if a scheduled backup is due.
	 *
	 * @param array $settings Plugin settings.
	 * @return bool
	 */
	private function vlwp_is_schedule_due( $settings ) {
		$last_run = (int) get_option( self::OPTION_LAST_SCHEDULED_RUN, 0 );
		if ( $last_run <= 0 ) {
			return true;
		}

		$interval_seconds = $this->vlwp_get_schedule_interval_seconds( $settings );

		return ( time() - $last_run ) >= $interval_seconds;
	}

	/**
	 * Resolve schedule interval in seconds.
	 *
	 * @param array $settings Plugin settings.
	 * @return int
	 */
	private function vlwp_get_schedule_interval_seconds( $settings ) {
		$value = max( 1, (int) ( $settings['schedule_interval_value'] ?? 1 ) );
		$unit = isset( $settings['schedule_interval_unit'] ) ? (string) $settings['schedule_interval_unit'] : 'hours';

		if ( 'minutes' === $unit ) {
			return $value * MINUTE_IN_SECONDS;
		}

		if ( 'days' === $unit ) {
			return $value * DAY_IN_SECONDS;
		}

		return $value * HOUR_IN_SECONDS;
	}

	/**
	 * Apply backup retention settings.
	 */
	private function vlwp_apply_retention_policy() {
		$settings = $this->vlwp_get_settings();
		if ( empty( $settings['retention_enabled'] ) ) {
			return;
		}

		$max_days = max( 1, (int) ( $settings['retention_days'] ?? 30 ) );
		$max_backups = max( 1, (int) ( $settings['retention_max_backups'] ?? 20 ) );

		// Use local index for retention decisions (includes metadata and checksum)
		$index = VLWP_Backup_Storage::vlwp_get_local_index();
		if ( empty( $index ) ) {
			return;
		}

		// sort by mtime desc
		$items = $index;
		$items = array_values( $items );
		usort(
			$items,
			static function ( $left, $right ) {
				return (int) $right['mtime'] - (int) $left['mtime'];
			}
		);

		$deleted = 0;
		$threshold = time() - ( $max_days * DAY_IN_SECONDS );

		foreach ( $items as $idx => $backup ) {
			$remove = false;

			if ( (int) $backup['mtime'] < $threshold ) {
				$remove = true;
			}

			if ( ! $remove && $idx >= $max_backups ) {
				$remove = true;
			}

			if ( ! $remove ) {
				continue;
			}

			$result = VLWP_Backup_Storage::vlwp_delete_local_backup( $backup['name'] );
			if ( ! empty( $result['success'] ) ) {
				$deleted++;
				// Also delete any remote index entries linked to this local backup
				$remote_index = VLWP_Backup_Storage::vlwp_get_remote_index();
				foreach ( $remote_index as $remote_name => $meta ) {
					if ( isset( $meta['local_name'] ) && $meta['local_name'] === $backup['name'] ) {
						$del = VLWP_Backup_Engine::vlwp_delete_remote_backup( $remote_name, $this->vlwp_get_settings() );
						if ( ! empty( $del['success'] ) ) {
							VLWP_Backup_Storage::vlwp_delete_remote_index_entry( $remote_name );
						}
					}
				}
			}
		}

		if ( $deleted > 0 ) {
			VLWP_Backup_Logger::vlwp_log( 'info', 'Local retention cleanup finished.', array( 'deleted' => $deleted ) );
		}

		if ( in_array( $settings['storage_mode'], array( 'ftp', 'ftps', 'sftp' ), true ) ) {
			// Remote retention using remote index and dedicated settings
			if ( empty( $settings['remote_retention_enabled'] ) ) {
				return;
			}

			$remote_index = VLWP_Backup_Storage::vlwp_get_remote_index();
			if ( empty( $remote_index ) ) {
				return;
			}

			// Prepare items sorted by mtime desc
			$items = array_values( $remote_index );
			usort(
				$items,
				static function ( $a, $b ) {
					return (int) $b['mtime'] - (int) $a['mtime'];
				}
			);

			$remote_deleted = 0;
			$remote_threshold = time() - ( max( 1, (int) $settings['remote_retention_days'] ) * DAY_IN_SECONDS );
			$remote_max = max( 1, (int) $settings['remote_retention_max_backups'] );

			foreach ( $items as $idx => $meta ) {
				$remove = false;

				if ( ! empty( $meta['mtime'] ) && (int) $meta['mtime'] < $remote_threshold ) {
					$remove = true;
				}

				if ( ! $remove && $idx >= $remote_max ) {
					$remove = true;
				}

				if ( ! $remove ) {
					continue;
				}

				$result = VLWP_Backup_Engine::vlwp_delete_remote_backup( $meta['name'], $settings );
				if ( ! empty( $result['success'] ) ) {
					$remote_deleted++;
					VLWP_Backup_Storage::vlwp_delete_remote_index_entry( $meta['name'] );
				}
			}

			if ( $remote_deleted > 0 ) {
				VLWP_Backup_Logger::vlwp_log( 'info', 'Remote retention cleanup finished.', array( 'deleted' => $remote_deleted ) );
			}
		}
	}

	/**
	 * Send notification email when enabled.
	 *
	 * @param array $result Backup result payload.
	 * @param array $settings Plugin settings.
	 */
	private function vlwp_maybe_send_backup_notification( $result, $settings ) {
		if ( empty( $settings['email_notification_enabled'] ) ) {
			return;
		}

		$to = isset( $settings['notification_email'] ) ? sanitize_email( $settings['notification_email'] ) : '';
		if ( '' === $to ) {
			return;
		}

		$subject = ! empty( $result['success'] )
			? __( 'Backup completed', 'vlwp-backup' )
			: __( 'Backup failed', 'vlwp-backup' );

		// Build a richer message with useful backup details
		$attachments = array();
		$details = array();

		$created_file = isset( $result['created_file'] ) ? (string) $result['created_file'] : '';
		if ( '' !== $created_file && file_exists( $created_file ) ) {
			$basename = basename( $created_file );
			$size = @filesize( $created_file );
			$details[] = sprintf( __( 'Backup: %s', 'vlwp-backup' ), $basename );
			if ( function_exists( 'size_format' ) ) {
				$details[] = sprintf( __( 'Size: %s', 'vlwp-backup' ), size_format( $size ) );
			} else {
				$details[] = sprintf( __( 'Size: %d bytes', 'vlwp-backup' ), $size );
			}
			$details[] = sprintf( __( 'Created at: %s', 'vlwp-backup' ), date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), @filemtime( $created_file ) ) );
			if ( isset( $result['checksum'] ) && '' !== $result['checksum'] ) {
				$details[] = sprintf( __( 'Checksum (SHA256): %s', 'vlwp-backup' ), $result['checksum'] );
			}
		} elseif ( isset( $result['file_name'] ) && '' !== $result['file_name'] ) {
			$details[] = sprintf( __( 'Backup name: %s', 'vlwp-backup' ), (string) $result['file_name'] );
		}

		// External storage info
		$external_stored = ! empty( $result['remote_file'] );
		$details[] = sprintf( __( 'Stored externally: %s', 'vlwp-backup' ), $external_stored ? __( 'Yes', 'vlwp-backup' ) : __( 'No', 'vlwp-backup' ) );
		if ( $external_stored ) {
			$details[] = sprintf( __( 'Remote path: /%s', 'vlwp-backup' ), (string) $result['remote_file'] );
		}

		// Compose message text
		$base_message = isset( $result['message'] ) ? (string) $result['message'] : '';
		if ( ! empty( $details ) ) {
			$message = $base_message . "\n\n" . implode( "\n", $details );
		} else {
			$message = $base_message;
		}

		if ( ! empty( $settings['email_attach_backup'] ) && ! empty( $result['created_file'] ) && file_exists( $result['created_file'] ) ) {
			$attachments[] = $result['created_file'];
		}

		wp_mail( $to, $subject, $message, array(), $attachments );
	}

	/**
	 * Try to acquire a lock.
	 *
	 * @param string $option_key Lock option key.
	 * @param string $context Optional lock context.
	 * @return bool
	 */
	private function vlwp_acquire_lock( $option_key, $context = '' ) {
		$now = time();
		$payload = array(
			'time' => $now,
			'context' => (string) $context,
		);

		if ( add_option( $option_key, $payload, '', false ) ) {
			return true;
		}

		$current = get_option( $option_key, array() );
		$current_time = is_array( $current ) && isset( $current['time'] ) ? (int) $current['time'] : 0;
		if ( $current_time > 0 && ( $now - $current_time ) < self::LOCK_TIMEOUT ) {
			return false;
		}

		update_option( $option_key, $payload, false );

		return true;
	}

	/**
	 * Release a lock option.
	 *
	 * @param string $option_key Lock option key.
	 */
	private function vlwp_release_lock( $option_key ) {
		delete_option( $option_key );
	}
}
