<?php
/**
 * Admin UI and settings.
 */

defined( 'ABSPATH' ) || exit;

		class VLWP_Backup_Admin {
			/**
			 * Singleton instance.
			 *
			 * @var VLWP_Backup_Admin|null
			 */
			private static $instance = null;

			/**
			 * Settings page slug.
			 *
			 * @var string
			 */
			const PAGE_SLUG = 'vlwp-backup';

			/**
			 * Report transient key prefix.
			 *
			 * @var string
			 */
			const REPORT_TRANSIENT_PREFIX = 'vlwp_backup_report_';

			/**
			 * Get singleton instance.
			 *
			 * @return VLWP_Backup_Admin
			 */
			public static function vlwp_get_instance() {
				if ( null === self::$instance ) {
					self::$instance = new self();
				}

				return self::$instance;
			}

			/**
			 * Register admin hooks.
			 */
			public function vlwp_init() {
				add_action( 'admin_menu', array( $this, 'vlwp_register_menu' ) );
				add_action( 'admin_init', array( $this, 'vlwp_register_settings' ) );
				// Dashboard widget and background status check
				add_action( 'wp_dashboard_setup', array( $this, 'vlwp_register_dashboard_widget' ) );
				add_action( 'vlwp_backup_check_status_cron', array( $this, 'vlwp_check_last_backup_status' ) );
				add_action( 'vlwp_backup_check_external_status_cron', array( $this, 'vlwp_check_external_status' ) );
				if ( ! wp_next_scheduled( 'vlwp_backup_check_status_cron' ) ) {
					wp_schedule_event( time(), 'hourly', 'vlwp_backup_check_status_cron' );
				}
				if ( ! wp_next_scheduled( 'vlwp_backup_check_external_status_cron' ) ) {
					wp_schedule_event( time(), 'hourly', 'vlwp_backup_check_external_status_cron' );
				}

				// AJAX handler for toggling auto-backups
				add_action( 'wp_ajax_vlwp_toggle_auto_backups', array( $this, 'vlwp_toggle_auto_backups' ) );

				// Manual dashboard status check (triggered from dashboard widget)
				add_action( 'admin_post_vlwp_backup_run_status_check', array( $this, 'vlwp_handle_run_status_check' ) );
				add_action( 'admin_post_vlwp_backup_run_manual', array( $this, 'vlwp_handle_manual_backup' ) );
				add_action( 'admin_post_vlwp_backup_run_test', array( $this, 'vlwp_handle_test_backup' ) );
				add_action( 'admin_post_vlwp_backup_test_remote', array( $this, 'vlwp_handle_test_remote_connection' ) );
				add_action( 'admin_post_vlwp_backup_upload_local', array( $this, 'vlwp_handle_upload_local_backup' ) );
				add_action( 'admin_post_vlwp_backup_download_local', array( $this, 'vlwp_handle_download_local_backup' ) );
				add_action( 'admin_post_vlwp_backup_delete_local', array( $this, 'vlwp_handle_delete_local_backup' ) );
				add_action( 'admin_post_vlwp_backup_download_remote', array( $this, 'vlwp_handle_download_remote_backup' ) );
				add_action( 'admin_post_vlwp_backup_delete_remote', array( $this, 'vlwp_handle_delete_remote_backup' ) );
				add_action( 'admin_post_vlwp_backup_validate_local', array( $this, 'vlwp_handle_validate_local_backup' ) );
				add_action( 'admin_post_vlwp_backup_validate_remote', array( $this, 'vlwp_handle_validate_remote_backup' ) );
        add_action( 'admin_post_vlwp_backup_restore_local', array( $this, 'vlwp_handle_restore_local_backup' ) );
        add_action( 'admin_post_vlwp_backup_run_external_check', array( $this, 'vlwp_handle_run_external_check' ) );
			}

      /**
       * Check last backup status and store result.
       */
      public function vlwp_check_last_backup_status() {
		$index = VLWP_Backup_Storage::vlwp_get_local_backups();
        $ok = is_array($index) && count($index) > 0;
        $checked_at = time();
		// Persist the last status in an option (no transient usage)
		update_option( 'vlwp_backup_status', array( 'ok' => $ok, 'checked_at' => $checked_at ), false );
      }

        /**
      * Check external storage accessibility and store status.
      */
      public function vlwp_check_external_status() {
        $settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
        $mode = isset($settings['storage_mode']) ? $settings['storage_mode'] : 'local';
        $ok = false;
        $checked_at = time();
        if (in_array($mode, array('ftp', 'ftps', 'sftp'), true)) {
          $result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_test_remote_connection();
          $ok = !empty($result['success']);
        }
		// Persist the external status in an option (no transient usage)
		update_option( 'vlwp_backup_external_status', array( 'ok' => $ok, 'checked_at' => $checked_at ), false );
      }

      /**
       * Toggle auto-backups (AJAX handler).
       */
      public function vlwp_toggle_auto_backups() {
        if ( ! current_user_can( 'manage_options' ) ) {
          wp_send_json_error( array( 'message' => __( 'Not allowed.', 'vlwp-backup' ) ) );
        }
        check_ajax_referer( 'vlwp_toggle_auto_backups', 'nonce' );
        $settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
        $enabled = empty( $settings['schedule_enabled'] ) ? 0 : 1;
        $settings['schedule_enabled'] = $enabled ? 0 : 1;

        // If enabling set to 24h default
        $settings['schedule_interval_value'] = 24;
        $settings['schedule_interval_unit'] = 'hours';

        update_option( vlwp_backup_option_key, $settings );
        wp_send_json_success( array( 'enabled' => (int) $settings['schedule_enabled'] ) );
      }

			/**
			 * Register top-level menu.
			 */
			public function vlwp_register_menu() {
				add_menu_page(
					__( 'Backup', 'vlwp-backup' ),
					__( 'Backup', 'vlwp-backup' ),
					'manage_options',
					self::PAGE_SLUG,
					array( $this, 'vlwp_render_list_page' ),
					'dashicons-database-export',
					59
				);

				add_submenu_page(
					self::PAGE_SLUG,
					__( 'List', 'vlwp-backup' ),
					__( 'List', 'vlwp-backup' ),
					'manage_options',
					self::PAGE_SLUG,
					array( $this, 'vlwp_render_list_page' )
				);

				add_submenu_page(
					self::PAGE_SLUG,
					__( 'Settings', 'vlwp-backup' ),
					__( 'Settings', 'vlwp-backup' ),
					'manage_options',
					self::PAGE_SLUG . '_settings',
					array( $this, 'vlwp_render_settings_page' )
				);
			}

			/**
			 * Register plugin settings.
			 */
			public function vlwp_register_settings() {
				register_setting( self::PAGE_SLUG, vlwp_backup_option_key, array( $this, 'vlwp_sanitize_settings' ) );

				add_settings_section(
					'vlwp_backup_section_general',
					__( 'General', 'vlwp-backup' ),
					'__return_false',
					self::PAGE_SLUG
				);

				add_settings_section(
					'vlwp_backup_section_schedule',
					__( 'Schedule', 'vlwp-backup' ),
					'__return_false',
					self::PAGE_SLUG
				);

				add_settings_section(
					'vlwp_backup_section_storage',
					__( 'Storage', 'vlwp-backup' ),
					'__return_false',
					self::PAGE_SLUG
				);

				add_settings_section(
					'vlwp_backup_section_notifications',
					__( 'Notifications', 'vlwp-backup' ),
					'__return_false',
					self::PAGE_SLUG
				);

				add_settings_section(
					'vlwp_backup_section_retention',
					__( 'Retention & Logging', 'vlwp-backup' ),
					'__return_false',
					self::PAGE_SLUG
				);

				$fields = array(
					'backup_password' => array( 'type' => 'password', 'label' => __( 'Backup password', 'vlwp-backup' ) ),
					'excluded_paths' => array( 'type' => 'textarea', 'label' => __( 'Excluded paths', 'vlwp-backup' ) ),
					'excluded_tables' => array( 'type' => 'textarea', 'label' => __( 'Excluded DB tables', 'vlwp-backup' ) ),
					'schedule_enabled' => array( 'type' => 'checkbox', 'label' => __( 'Enable schedule', 'vlwp-backup' ) ),
					'schedule_interval_value' => array( 'type' => 'number', 'label' => __( 'Schedule interval value', 'vlwp-backup' ) ),
					'schedule_interval_unit' => array( 'type' => 'select', 'label' => __( 'Schedule interval unit', 'vlwp-backup' ) ),
					'storage_mode' => array( 'type' => 'select', 'label' => __( 'Storage mode', 'vlwp-backup' ) ),
					'ftp_host' => array( 'type' => 'text', 'label' => __( 'FTP host', 'vlwp-backup' ) ),
					'ftp_port' => array( 'type' => 'number', 'label' => __( 'FTP port', 'vlwp-backup' ) ),
					'ftp_username' => array( 'type' => 'text', 'label' => __( 'FTP username', 'vlwp-backup' ) ),
					'ftp_password' => array( 'type' => 'password', 'label' => __( 'FTP password', 'vlwp-backup' ) ),
					'ftp_path' => array( 'type' => 'text', 'label' => __( 'FTP path', 'vlwp-backup' ) ),
					'ftp_ssl' => array( 'type' => 'checkbox', 'label' => __( 'Use FTP over SSL', 'vlwp-backup' ) ),
					'email_notification_enabled' => array( 'type' => 'checkbox', 'label' => __( 'Enable email notifications', 'vlwp-backup' ) ),
					'email_attach_backup' => array( 'type' => 'checkbox', 'label' => __( 'Attach backup in email', 'vlwp-backup' ) ),
					'notification_email' => array( 'type' => 'text', 'label' => __( 'Notification email', 'vlwp-backup' ) ),
					'retention_enabled' => array( 'type' => 'checkbox', 'label' => __( 'Enable retention cleanup', 'vlwp-backup' ) ),
					'retention_days' => array( 'type' => 'number', 'label' => __( 'Retention days', 'vlwp-backup' ) ),
					'retention_max_backups' => array( 'type' => 'number', 'label' => __( 'Max backups', 'vlwp-backup' ) ),
					'remote_retention_enabled' => array( 'type' => 'checkbox', 'label' => __( 'Enable external retention cleanup', 'vlwp-backup' ) ),
					'remote_retention_days' => array( 'type' => 'number', 'label' => __( 'External retention days', 'vlwp-backup' ) ),
					'remote_retention_max_backups' => array( 'type' => 'number', 'label' => __( 'External max backups', 'vlwp-backup' ) ),
					'enable_logging' => array( 'type' => 'checkbox', 'label' => __( 'Enable logging', 'vlwp-backup' ) ),
					'log_file_path' => array( 'type' => 'text', 'label' => __( 'Log file path', 'vlwp-backup' ) ),
					'multisite_scope' => array( 'type' => 'select', 'label' => __( 'Multisite scope', 'vlwp-backup' ) ),
					'delete_linked_remote_on_local_delete' => array( 'type' => 'checkbox', 'label' => __( 'Delete linked external copy when deleting local', 'vlwp-backup' ) ),
				);



				$sections = array(
					'backup_password' => 'vlwp_backup_section_general',
					'excluded_paths' => 'vlwp_backup_section_general',
					'excluded_tables' => 'vlwp_backup_section_general',
					'schedule_enabled' => 'vlwp_backup_section_schedule',
					'schedule_interval_value' => 'vlwp_backup_section_schedule',
					'schedule_interval_unit' => 'vlwp_backup_section_schedule',
					'storage_mode' => 'vlwp_backup_section_storage',
					'ftp_host' => 'vlwp_backup_section_storage',
					'ftp_port' => 'vlwp_backup_section_storage',
					'ftp_username' => 'vlwp_backup_section_storage',
					'ftp_password' => 'vlwp_backup_section_storage',
					'ftp_path' => 'vlwp_backup_section_storage',
					'ftp_ssl' => 'vlwp_backup_section_storage',
					'email_notification_enabled' => 'vlwp_backup_section_notifications',
					'email_attach_backup' => 'vlwp_backup_section_notifications',
					'notification_email' => 'vlwp_backup_section_notifications',
					'retention_enabled' => 'vlwp_backup_section_retention',
					'retention_days' => 'vlwp_backup_section_retention',
					'retention_max_backups' => 'vlwp_backup_section_retention',
					'remote_retention_enabled' => 'vlwp_backup_section_retention',
					'remote_retention_days' => 'vlwp_backup_section_retention',
					'remote_retention_max_backups' => 'vlwp_backup_section_retention',
					'enable_logging' => 'vlwp_backup_section_retention',
					'log_file_path' => 'vlwp_backup_section_retention',
					'multisite_scope' => 'vlwp_backup_section_retention',
					'delete_linked_remote_on_local_delete' => 'vlwp_backup_section_retention',
				);

				foreach ( $fields as $key => $field ) {
					add_settings_field(
						$key,
						$field['label'],
						array( $this, 'vlwp_render_field' ),
						self::PAGE_SLUG,
						$sections[ $key ],
						array(
							'key' => $key,
							'type' => $field['type'],
						)
					);
				}
			}

			/**
			 * Sanitize settings payload.
			 *
			 * @param array $raw Raw option value.
			 * @return array
			 */
			public function vlwp_sanitize_settings( $raw ) {
				$defaults = VLWP_Backup_Core::vlwp_get_default_settings();
				$raw = is_array( $raw ) ? $raw : array();

				$clean = $defaults;
				$clean['schedule_enabled'] = empty( $raw['schedule_enabled'] ) ? 0 : 1;
				$clean['schedule_interval_value'] = max( 1, (int) ( $raw['schedule_interval_value'] ?? 1 ) );
				$clean['schedule_interval_unit'] = in_array( $raw['schedule_interval_unit'] ?? '', array( 'minutes', 'hours', 'days' ), true ) ? $raw['schedule_interval_unit'] : 'hours';
				$clean['storage_mode'] = in_array( $raw['storage_mode'] ?? '', array( 'local', 'ftp', 'ftps', 'sftp' ), true ) ? $raw['storage_mode'] : 'local';
				$clean['ftp_host'] = sanitize_text_field( $raw['ftp_host'] ?? '' );
				$clean['ftp_port'] = max( 1, (int) ( $raw['ftp_port'] ?? 21 ) );
				$clean['ftp_username'] = sanitize_text_field( $raw['ftp_username'] ?? '' );
				$clean['ftp_password'] = sanitize_text_field( $raw['ftp_password'] ?? '' );
				$clean['ftp_path'] = sanitize_text_field( $raw['ftp_path'] ?? '' );
				$clean['ftp_ssl'] = empty( $raw['ftp_ssl'] ) ? 0 : 1;
				$clean['backup_password'] = sanitize_text_field( $raw['backup_password'] ?? '' );
				$clean['email_notification_enabled'] = empty( $raw['email_notification_enabled'] ) ? 0 : 1;
				$clean['email_attach_backup'] = empty( $raw['email_attach_backup'] ) ? 0 : 1;
				$clean['notification_email'] = sanitize_email( $raw['notification_email'] ?? '' );
				$clean['excluded_paths'] = sanitize_textarea_field( $raw['excluded_paths'] ?? '' );
				$clean['excluded_tables'] = sanitize_textarea_field( $raw['excluded_tables'] ?? '' );
				$clean['retention_enabled'] = empty( $raw['retention_enabled'] ) ? 0 : 1;
				$clean['retention_days'] = max( 1, (int) ( $raw['retention_days'] ?? 30 ) );
				$clean['retention_max_backups'] = max( 1, (int) ( $raw['retention_max_backups'] ?? 20 ) );
				$clean['enable_logging'] = empty( $raw['enable_logging'] ) ? 0 : 1;
				$clean['log_file_path'] = sanitize_text_field( $raw['log_file_path'] ?? '' );
				$clean['multisite_scope'] = in_array( $raw['multisite_scope'] ?? '', array( 'site', 'network' ), true ) ? $raw['multisite_scope'] : 'site';
				$clean['delete_linked_remote_on_local_delete'] = empty( $raw['delete_linked_remote_on_local_delete'] ) ? 0 : 1;
				$clean['remote_retention_enabled'] = empty( $raw['remote_retention_enabled'] ) ? 0 : 1;
				$clean['remote_retention_days'] = max( 1, (int) ( $raw['remote_retention_days'] ?? 30 ) );
				$clean['remote_retention_max_backups'] = max( 1, (int) ( $raw['remote_retention_max_backups'] ?? 20 ) );

				return $clean;
			}

			/**
			 * Render generic field.
			 *
			 * @param array $args Field args.
			 */
			public function vlwp_render_field( $args ) {
				$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
				$key = isset( $args['key'] ) ? (string) $args['key'] : '';
				$type = isset( $args['type'] ) ? (string) $args['type'] : 'text';
				$value = isset( $settings[ $key ] ) ? $settings[ $key ] : '';
				$field_name = vlwp_backup_option_key . '[' . $key . ']';


				$external_keys = array( 'ftp_host', 'ftp_port', 'ftp_username', 'ftp_password', 'ftp_path', 'ftp_ssl' );
				$mode = isset( $settings['storage_mode'] ) ? (string) $settings['storage_mode'] : 'local';

				   // Always render a wrapper <li> for external fields so JS can toggle them live
				   if ( in_array( $key, $external_keys, true ) ) {
					   // Map which fields are required for which modes
					   $field_modes = array(
						   'ftp_host' => array('ftp', 'ftps', 'sftp'),
						   'ftp_port' => array('ftp', 'ftps', 'sftp'),
						   'ftp_username' => array('ftp', 'ftps', 'sftp'),
						   'ftp_password' => array('ftp', 'ftps', 'sftp'),
						   'ftp_path' => array('ftp', 'ftps', 'sftp'),
						   'ftp_ssl' => array('ftp'),
					   );
					   $modes = isset($field_modes[$key]) ? $field_modes[$key] : array();
					   echo '<li class="vlwp-external-field" data-key="' . esc_attr($key) . '" data-modes="' . esc_attr(implode(',', $modes)) . '" style="display:none;">';
				   }

				   if ( 'checkbox' === $type ) {
					   $checked = checked( 1, (int) $value, false );
					   $disabled = '';
					   if ( $key === 'email_attach_backup' ) {
						   $password_set = ! empty( $settings['backup_password'] );
						   if ( ! $password_set ) {
							   $disabled = 'disabled="disabled"';
						   }
						   printf( '<input type="checkbox" name="%s" value="1" %s %s />', esc_attr( $field_name ), $checked, $disabled );
						   if (! $password_set) echo ' <span class="description">' . esc_html__( 'To use attached backups, password protection is required.', 'vlwp-backup' ) . '</span>';
						   if ( in_array( $key, $external_keys, true ) ) {
							   echo '</li>';
						   }
						   return;
					   }
					printf( '<input type="checkbox" name="%s" value="1" %s />', esc_attr( $field_name ), $checked );
					// description for checkbox fields
					$cb_descs = array(
						'schedule_enabled' => __( 'Enable scheduled automatic backups.', 'vlwp-backup' ),
						'retention_enabled' => __( 'When enabled, old backups will be cleaned up automatically.', 'vlwp-backup' ),
						'remote_retention_enabled' => __( 'Enable retention rules for external backups.', 'vlwp-backup' ),
						'enable_logging' => __( 'Log plugin activity for debugging.', 'vlwp-backup' ),
						'delete_linked_remote_on_local_delete' => __( 'Also delete the matching external copy when you remove a local backup.', 'vlwp-backup' ),
					);
					if ( isset( $cb_descs[ $key ] ) ) {
						echo ' <span class="description">' . esc_html( $cb_descs[ $key ] ) . '</span>';
					}
					if ( in_array( $key, $external_keys, true ) ) {
						echo '</li>';
					}
					return;
				   }

				   if ( 'textarea' === $type ) {
					   printf( '<textarea name="%s" rows="4" cols="50" class="large-text">%s</textarea>', esc_attr( $field_name ), esc_textarea( (string) $value ) );
					$ta_descs = array(
						'excluded_paths' => __( 'One path per line. Prefixes are relative to the WP root.', 'vlwp-backup' ),
						'excluded_tables' => __( 'One table name per line (without prefix).', 'vlwp-backup' ),
					);
					if ( isset( $ta_descs[ $key ] ) ) {
						echo ' <p class="description">' . esc_html( $ta_descs[ $key ] ) . '</p>';
					}
					if ( in_array( $key, $external_keys, true ) ) {
						echo '</li>';
					}
					return;
				   }

				   if ( 'select' === $type ) {
					   $options = array();
					   if ( 'schedule_interval_unit' === $key ) {
						   $options = array(
							   'minutes' => __( 'Minutes', 'vlwp-backup' ),
							   'hours' => __( 'Hours', 'vlwp-backup' ),
							   'days' => __( 'Days', 'vlwp-backup' ),
						   );
					   } elseif ( 'storage_mode' === $key ) {
						   $options = array(
							   'local' => __( 'Local only', 'vlwp-backup' ),
							   'ftp' => __( 'Local + FTP', 'vlwp-backup' ),
							   'ftps' => __( 'Local + FTPS', 'vlwp-backup' ),
							   'sftp' => __( 'Local + SFTP', 'vlwp-backup' ),
						   );
					   } elseif ( 'multisite_scope' === $key ) {
						   $options = array(
							   'site' => __( 'Current site', 'vlwp-backup' ),
							   'network' => __( 'Whole network', 'vlwp-backup' ),
						   );
					   }
					   echo '<select name="' . esc_attr( $field_name ) . '" id="vlwp-field-' . esc_attr( $key ) . '">';
					   foreach ( $options as $option_key => $label ) {
						   echo '<option value="' . esc_attr( $option_key ) . '"' . selected( (string) $value, (string) $option_key, false ) . '>' . esc_html( $label ) . '</option>';
					   }
					   echo '</select>';
					$sel_descs = array(
						'schedule_interval_unit' => __( 'Choose the time unit for the schedule interval.', 'vlwp-backup' ),
						'storage_mode' => __( 'Select storage target for backups.', 'vlwp-backup' ),
						'multisite_scope' => __( 'Scope to backup in multisite installations.', 'vlwp-backup' ),
					);
					if ( isset( $sel_descs[ $key ] ) ) {
						echo ' <p class="description">' . esc_html( $sel_descs[ $key ] ) . '</p>';
					}
					if ( in_array( $key, $external_keys, true ) ) {
						echo '</li>';
					}
					return;
				}

				  $input_type = in_array( $type, array( 'text', 'password', 'number' ), true ) ? $type : 'text';
				// Short descriptive hints for fields
				$descriptions = array(
					'backup_password' => __( 'Password used to encrypt backups. Required to attach backups to emails securely.', 'vlwp-backup' ),
					'excluded_paths' => __( 'Enter one path per line (relative to WP root). Matches are excluded from file backups.', 'vlwp-backup' ),
					'excluded_tables' => __( 'Enter one table name per line (no prefix) to exclude from DB export.', 'vlwp-backup' ),
					'schedule_interval_value' => __( 'Number of units between scheduled backups (e.g. 24).', 'vlwp-backup' ),
					'schedule_interval_unit' => __( 'Unit for the schedule interval (minutes/hours/days).', 'vlwp-backup' ),
					'storage_mode' => __( 'Choose where copies are stored. FTP/SFTP require providing credentials below.', 'vlwp-backup' ),
					'ftp_host' => __( 'Hostname or IP of remote storage server.', 'vlwp-backup' ),
					'ftp_port' => __( 'Remote server port (default: 21 for FTP, 22 for SFTP).', 'vlwp-backup' ),
					'ftp_username' => __( 'Username for the remote storage account.', 'vlwp-backup' ),
					'ftp_password' => __( 'Password for the remote storage account.', 'vlwp-backup' ),
					'ftp_path' => __( 'Remote directory path where backups will be stored.', 'vlwp-backup' ),
					'ftp_ssl' => __( 'Enable FTP over SSL (FTPS). Use when your server supports it.', 'vlwp-backup' ),
					'notification_email' => __( 'Email address to receive backup notifications.', 'vlwp-backup' ),
					'retention_days' => __( 'Remove backups older than this many days.', 'vlwp-backup' ),
					'retention_max_backups' => __( 'Maximum number of backups to keep. Older ones are deleted.', 'vlwp-backup' ),
					'remote_retention_days' => __( 'Same as above, but for external backups.', 'vlwp-backup' ),
					'remote_retention_max_backups' => __( 'Max number of external backups to retain.', 'vlwp-backup' ),
					'enable_logging' => __( 'Write plugin actions to a log file for troubleshooting.', 'vlwp-backup' ),
					'log_file_path' => __( 'Path to store the plugin log file (absolute or relative to WP root).', 'vlwp-backup' ),
					'multisite_scope' => __( 'Choose whether backups include the current site only or the entire network.', 'vlwp-backup' ),
					'delete_linked_remote_on_local_delete' => __( 'When enabled, deleting a local backup also removes its linked external copy (if any).', 'vlwp-backup' ),
				);

				echo '<input type="' . esc_attr( $input_type ) . '" class="regular-text" name="' . esc_attr( $field_name ) . '" value="' . esc_attr( (string) $value ) . '" />';
				if ( isset( $descriptions[ $key ] ) ) {
					echo ' <p class="description">' . esc_html( $descriptions[ $key ] ) . '</p>';
				}
				if ( in_array( $key, $external_keys, true ) ) {
					echo '</li>';
				}
			}

			/**
			 * Manual backup action.
			 */
			public function vlwp_handle_manual_backup() {
				if ( ! current_user_can( 'manage_options' ) ) {
					wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
				}

				check_admin_referer( 'vlwp_backup_run_manual' );

				$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_run_backup(
					array(
						'origin' => 'manual',
					)
				);

				$this->vlwp_redirect_with_notice( $result );
			}

	/**
	 * Upload local backup action.
	 */
	public function vlwp_handle_upload_local_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		check_admin_referer( 'vlwp_backup_upload_local' );

		$uploaded_file = isset( $_FILES['vlwp_backup_upload_file'] ) ? wp_unslash( $_FILES['vlwp_backup_upload_file'] ) : array();
		$result = VLWP_Backup_Storage::vlwp_store_uploaded_backup( $uploaded_file );

		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Test backup action.
	 */
	public function vlwp_handle_test_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		check_admin_referer( 'vlwp_backup_run_test' );

		$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_run_backup_restore_test();
		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Test external storage connection action.
	 */
	public function vlwp_handle_test_remote_connection() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		check_admin_referer( 'vlwp_backup_test_remote' );

		$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_test_remote_connection();
		$this->vlwp_redirect_with_notice( $result );
	}

	// Indexing/rescan handlers removed to simplify functionality.

	/**
	 * Download local backup action.
	 */
	public function vlwp_handle_download_local_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_download_local_' . $file_name );

		$path = VLWP_Backup_Storage::vlwp_get_local_backup_path_by_name( $file_name );
		if ( '' === $path ) {
			$this->vlwp_redirect_with_notice(
				array(
					'success' => false,
					'message' => __( 'Backup file was not found.', 'vlwp-backup' ),
				)
			);
		}

		nocache_headers();
		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: application/zip' );
		header( 'Content-Disposition: attachment; filename="' . basename( $path ) . '"' );
		header( 'Content-Length: ' . (string) filesize( $path ) );
		readfile( $path );
		exit;
	}

	/**
	 * Delete local backup action.
	 */
	public function vlwp_handle_delete_local_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_delete_local_' . $file_name );

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();

		// Removing automatic deletion of linked remote copies (indexing removed).

		$result = VLWP_Backup_Storage::vlwp_delete_local_backup( $file_name );
		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Validate local backup action.
	 */
	public function vlwp_handle_validate_local_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_validate_local_' . $file_name );

		$path = VLWP_Backup_Storage::vlwp_get_local_backup_path_by_name( $file_name );
		if ( '' === $path ) {
			$this->vlwp_redirect_with_notice(
				array(
					'success' => false,
					'message' => __( 'Backup file was not found.', 'vlwp-backup' ),
				)
			);
		}

		$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_validate_restore_dry_run( $path );
		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Download remote backup action.
	 */
	public function vlwp_handle_download_remote_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_download_remote_' . $file_name );

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		$temp_file = trailingslashit( VLWP_Backup_Storage::vlwp_get_local_backup_dir() ) . 'tmp-download-' . wp_generate_password( 8, false, false ) . '-' . $file_name;

		$download = VLWP_Backup_Engine::vlwp_download_remote_backup( $file_name, $temp_file, $settings );
		if ( empty( $download['success'] ) ) {
			$this->vlwp_redirect_with_notice( $download );
		}

		nocache_headers();
		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: application/zip' );
		header( 'Content-Disposition: attachment; filename="' . basename( $file_name ) . '"' );
		header( 'Content-Length: ' . (string) filesize( $temp_file ) );
		readfile( $temp_file );
		@unlink( $temp_file );
		exit;
	}

	/**
	 * Delete remote backup action.
	 */
	public function vlwp_handle_delete_remote_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_delete_remote_' . $file_name );

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		$result = VLWP_Backup_Engine::vlwp_delete_remote_backup( $file_name, $settings );
		if ( ! empty( $result['success'] ) ) {
			VLWP_Backup_Storage::vlwp_delete_remote_index_entry( $file_name );
		}
		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Validate remote backup action.
	 */
	public function vlwp_handle_validate_remote_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		$file_name = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
		check_admin_referer( 'vlwp_backup_validate_remote_' . $file_name );

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		$temp_file = trailingslashit( VLWP_Backup_Storage::vlwp_get_local_backup_dir() ) . 'tmp-validate-' . wp_generate_password( 8, false, false ) . '-' . $file_name;

		$download = VLWP_Backup_Engine::vlwp_download_remote_backup( $file_name, $temp_file, $settings );
		if ( empty( $download['success'] ) ) {
			$this->vlwp_redirect_with_notice( $download );
		}

		$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_validate_restore_dry_run( $temp_file );

		if ( file_exists( $temp_file ) ) {
			@unlink( $temp_file );
		}

		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Restore local backup action.
	 */
	public function vlwp_handle_restore_local_backup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
		}

		check_admin_referer( 'vlwp_backup_restore_local' );

		$file_name = isset( $_POST['file'] ) ? sanitize_file_name( wp_unslash( $_POST['file'] ) ) : '';
		$source = isset( $_POST['source'] ) ? sanitize_key( wp_unslash( $_POST['source'] ) ) : 'local';
		$restore_db = isset( $_POST['restore_db'] ) ? 1 : 0;
		$restore_wp_content = isset( $_POST['restore_wp_content'] ) ? 1 : 0;

		if ( ! $restore_db && ! $restore_wp_content ) {
			$this->vlwp_redirect_with_notice(
				array(
					'success' => false,
					'message' => __( 'Select at least one restore target.', 'vlwp-backup' ),
				)
			);
		}

		$path = '';
		$temp_remote_file = '';

		if ( 'remote' === $source ) {
			$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
			$temp_remote_file = trailingslashit( VLWP_Backup_Storage::vlwp_get_local_backup_dir() ) . 'tmp-restore-' . wp_generate_password( 8, false, false ) . '-' . $file_name;
			$download = VLWP_Backup_Engine::vlwp_download_remote_backup( $file_name, $temp_remote_file, $settings );

			if ( empty( $download['success'] ) ) {
				$this->vlwp_redirect_with_notice( $download );
			}

			$path = $temp_remote_file;
		} else {
			$path = VLWP_Backup_Storage::vlwp_get_local_backup_path_by_name( $file_name );
		}

		if ( '' === $path ) {
			$this->vlwp_redirect_with_notice(
				array(
					'success' => false,
					'message' => __( 'Backup file was not found.', 'vlwp-backup' ),
				)
			);
		}

		$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_restore_backup(
			$path,
			array(
				'restore_db' => (bool) $restore_db,
				'restore_wp_content' => (bool) $restore_wp_content,
			)
		);

		if ( '' !== $temp_remote_file && file_exists( $temp_remote_file ) ) {
			@unlink( $temp_remote_file );
		}

		$this->vlwp_redirect_with_notice( $result );
	}

	/**
	 * Redirect back to settings page with admin notice.
	 *
	 * @param array $result Operation result.
	 */
	private function vlwp_redirect_with_notice( $result ) {
		$status = ! empty( $result['success'] ) ? 'success' : 'error';
		$message = isset( $result['message'] ) ? (string) $result['message'] : '';
		$report_key = '';

		if ( isset( $result['report'] ) && is_array( $result['report'] ) ) {
			$report_key = wp_generate_password( 12, false, false );
			// Store report in an option instead of transient so it survives the redirect
			update_option( self::REPORT_TRANSIENT_PREFIX . $report_key, $result['report'], false );
		}

		$redirect = add_query_arg(
			array(
				'page' => self::PAGE_SLUG,
				'vlwp_backup_notice' => $status,
				'vlwp_backup_message' => $message,
				'vlwp_backup_report' => $report_key,
			),
			admin_url( 'admin.php' )
		);

		wp_safe_redirect( $redirect );
		exit;
	}

	/**
	 * Render admin page.
	 */
	public function vlwp_render_list_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$notice_type = isset( $_GET['vlwp_backup_notice'] ) ? sanitize_key( wp_unslash( $_GET['vlwp_backup_notice'] ) ) : '';
		$notice_message = isset( $_GET['vlwp_backup_message'] ) ? sanitize_text_field( wp_unslash( $_GET['vlwp_backup_message'] ) ) : '';
		$report_param = isset( $_GET['vlwp_backup_report'] ) ? preg_replace( '/[^a-zA-Z0-9]/', '', (string) wp_unslash( $_GET['vlwp_backup_report'] ) ) : '';
		$report = array();

		if ( '' !== $report_param ) {
			// Retrieve report from option storage (replacing transient usage)
			$report = get_option( self::REPORT_TRANSIENT_PREFIX . $report_param, array() );
			if ( ! is_array( $report ) ) {
				$report = array();
			}
			delete_option( self::REPORT_TRANSIENT_PREFIX . $report_param );
		}
		$restore_file = isset( $_GET['vlwp_restore_file'] ) ? sanitize_file_name( wp_unslash( $_GET['vlwp_restore_file'] ) ) : '';
		$restore_source = isset( $_GET['vlwp_restore_source'] ) ? sanitize_key( wp_unslash( $_GET['vlwp_restore_source'] ) ) : 'local';
		$restore_source = in_array( $restore_source, array( 'local', 'remote' ), true ) ? $restore_source : 'local';
		$restore_nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		$restore_nonce_action = 'remote' === $restore_source ? 'vlwp_backup_restore_options_remote_' . $restore_file : 'vlwp_backup_restore_options_' . $restore_file;
		$show_restore_form = ( '' !== $restore_file && wp_verify_nonce( $restore_nonce, $restore_nonce_action ) );

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		$show_remote = in_array( $settings['storage_mode'], array( 'ftp', 'ftps', 'sftp' ), true );
		// Build combined items
		$local_files = VLWP_Backup_Storage::vlwp_get_local_backups();
		$remote_items = $show_remote ? VLWP_Backup_Engine::vlwp_list_remote_backups( $settings ) : array();
		// Map remote index for linking and get local index metadata
		$remote_index = VLWP_Backup_Storage::vlwp_get_remote_index();
		$local_index = VLWP_Backup_Storage::vlwp_get_local_index();
		$local_index_map = array();
		foreach ( $remote_index as $rname => $meta ) {
			if ( ! empty( $meta['local_name'] ) ) {
				$local_index_map[ $meta['local_name'] ][] = $rname;
			}
		}

		$combined = array();
		foreach ( $local_files as $f ) {
			$name = isset( $f['name'] ) ? $f['name'] : '';
			$type = 'manual';
			if ( isset( $local_index[ $name ] ) && isset( $local_index[ $name ]['type'] ) ) {
				$type = $local_index[ $name ]['type'];
			}
			$combined[] = array(
				'name' => $name,
				'source' => 'local',
				'linked' => isset( $local_index_map[ $name ] ) ? $local_index_map[ $name ] : array(),
				'size' => isset( $f['size'] ) ? $f['size'] : 0,
				'mtime' => isset( $f['mtime'] ) ? $f['mtime'] : 0,
				'type' => $type,
			);
		}

		foreach ( $remote_items as $r ) {
			$rname = isset( $r['name'] ) ? $r['name'] : '';
			$rtype = 'manual';
			if ( isset( $remote_index[ $rname ] ) && isset( $remote_index[ $rname ]['type'] ) ) {
				$rtype = $remote_index[ $rname ]['type'];
			}
			$linked_to = isset( $r['local_name'] ) ? $r['local_name'] : '';
			if ( '' === $linked_to && isset( $remote_index[ $rname ] ) && isset( $remote_index[ $rname ]['local_name'] ) ) {
				$linked_to = $remote_index[ $rname ]['local_name'];
			}
			$combined[] = array(
				'name' => $rname,
				'source' => 'remote',
				'linked_to' => $linked_to,
				'size' => isset( $r['size'] ) ? $r['size'] : 0,
				'mtime' => isset( $r['mtime'] ) ? $r['mtime'] : 0,
				'type' => $rtype,
			);
		}

		// Apply filter param — default to showing only Local backups
		$filter = isset( $_GET['vlwp_filter'] ) ? sanitize_key( wp_unslash( $_GET['vlwp_filter'] ) ) : 'local';
		$filter = in_array( $filter, array( 'local', 'external' ), true ) ? $filter : 'local';
		$items_to_show = array();
		foreach ( $combined as $it ) {
			if ( 'all' === $filter ) {
				$items_to_show[] = $it;
				continue;
			}
			if ( 'local' === $filter && 'local' === $it['source'] ) {
				$items_to_show[] = $it;
			}
			if ( 'external' === $filter && 'remote' === $it['source'] ) {
				$items_to_show[] = $it;
			}
		}

		$list_table = new VLWP_Backup_Combined_List_Table( $items_to_show );
		$list_table->prepare_items();
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Backup', 'vlwp-backup' ); ?></h1>

			<?php if ( in_array( $notice_type, array( 'success', 'error' ), true ) && '' !== $notice_message ) : ?>
				<div class="notice notice-<?php echo esc_attr( $notice_type ); ?> is-dismissible">
					<p><?php echo esc_html( $notice_message ); ?></p>
					<?php if ( ! empty( $report ) ) : ?>
						<pre style="margin:8px 0 0;white-space:pre-wrap;"><?php echo esc_html( wp_json_encode( $report, JSON_PRETTY_PRINT ) ); ?></pre>
					<?php endif; ?>
				</div>
			<?php endif; ?>

			<hr />
			<h2><?php esc_html_e( 'Actions', 'vlwp-backup' ); ?></h2>
      <div style="display: flex; flex-wrap: wrap; gap: 10px;">
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
          <input type="hidden" name="action" value="vlwp_backup_run_manual" />
          <?php wp_nonce_field( 'vlwp_backup_run_manual' ); ?>
          <?php submit_button( __( 'Run backup now', 'vlwp-backup' ), 'secondary', 'submit', false ); ?>
        </form>
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
          <input type="hidden" name="action" value="vlwp_backup_run_test" />
          <?php wp_nonce_field( 'vlwp_backup_run_test' ); ?>
          <?php submit_button( __( 'Run backup test', 'vlwp-backup' ), 'secondary', 'submit', false ); ?>
        </form>
        <?php if ( $show_remote ) : ?>
          <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
            <input type="hidden" name="action" value="vlwp_backup_test_remote" />
            <?php wp_nonce_field( 'vlwp_backup_test_remote' ); ?>
            <?php submit_button( __( 'Test external connection', 'vlwp-backup' ), 'secondary', 'submit', false ); ?>
          </form>
        <?php endif; ?>
      </div>

			<h2><?php esc_html_e( 'Upload local backup', 'vlwp-backup' ); ?></h2>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
				<input type="hidden" name="action" value="vlwp_backup_upload_local" />
				<?php wp_nonce_field( 'vlwp_backup_upload_local' ); ?>
				<input type="file" name="vlwp_backup_upload_file" accept=".zip" required />
				<?php submit_button( __( 'Upload backup', 'vlwp-backup' ), 'secondary', 'submit', false ); ?>
			</form>
			<h2><?php esc_html_e( 'Backups', 'vlwp-backup' ); ?></h2>
			<?php
			// Filter links similar to post list views — default is Local
			$base_link = esc_url( add_query_arg( array( 'page' => self::PAGE_SLUG ), admin_url( 'admin.php' ) ) );
			$local_count = 0;
			$external_count = 0;
			foreach ( $combined as $c ) {
				if ( isset( $c['source'] ) && 'remote' === $c['source'] ) {
					$external_count++;
				} else {
					$local_count++;
				}
			}

			$current_filter = $filter;
			?>
			<p class="subsubsub" style="margin-top:6px;">
				<a href="<?php echo esc_url( add_query_arg( 'vlwp_filter', 'local', $base_link ) ); ?>" class="<?php echo $current_filter === 'local' ? 'current' : ''; ?>"><?php printf( esc_html__( 'Local (%d)', 'vlwp-backup' ), $local_count ); ?></a> |
				<a href="<?php echo esc_url( add_query_arg( 'vlwp_filter', 'external', $base_link ) ); ?>" class="<?php echo $current_filter === 'external' ? 'current' : ''; ?>"><?php printf( esc_html__( 'External (%d)', 'vlwp-backup' ), $external_count ); ?></a>
			</p>

			<div class="vlwp-backup-list" style="margin-top:12px;">
			<?php
			// Style: make the filename (first) column wider than others and ellipsize long names.
			echo '<style>.vlwp-backup-list .wp-list-table{table-layout:fixed;} .vlwp-backup-list .wp-list-table th.column-name, .vlwp-backup-list .wp-list-table td.column-name{width:45%;max-width:45%;} .vlwp-backup-list .wp-list-table td, .vlwp-backup-list .wp-list-table th{white-space:nowrap;overflow:hidden;text-overflow:ellipsis;} </style>';
			$list_table->display(); ?>
			</div>

			<?php if ( $show_restore_form ) : ?>
				<hr />
				<h2><?php esc_html_e( 'Restore options', 'vlwp-backup' ); ?></h2>
				<p><?php esc_html_e( 'Restoring will overwrite selected parts of the current site. Create a fresh backup first.', 'vlwp-backup' ); ?></p>
				<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
					<input type="hidden" name="action" value="vlwp_backup_restore_local" />
					<input type="hidden" name="file" value="<?php echo esc_attr( $restore_file ); ?>" />
					<input type="hidden" name="source" value="<?php echo esc_attr( $restore_source ); ?>" />
					<?php wp_nonce_field( 'vlwp_backup_restore_local' ); ?>
					<p><label><input type="checkbox" name="restore_db" value="1" checked /> <?php esc_html_e( 'Restore database', 'vlwp-backup' ); ?></label></p>
					<p><label><input type="checkbox" name="restore_wp_content" value="1" checked /> <?php esc_html_e( 'Restore wp-content files', 'vlwp-backup' ); ?></label></p>
					<?php submit_button( __( 'Start restore', 'vlwp-backup' ), 'primary', 'submit', false, array( 'onclick' => "return confirm('" . esc_js( __( 'Proceed with restore? This action cannot be undone.', 'vlwp-backup' ) ) . "');" ) ); ?>
				</form>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Render settings page.
	 */
	public function vlwp_render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Backup Settings', 'vlwp-backup' ); ?></h1>
			<?php
			// Merge remote connection test into the main settings notice when saving.
			$printed = false;
			$errors = get_settings_errors();
			if ( isset( $_GET['settings-updated'] ) ) {
				$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
				$main_notice = null;
				if ( in_array( $settings['storage_mode'] ?? 'local', array( 'ftp', 'ftps', 'sftp' ), true ) ) {
					$result = VLWP_Backup_Core::vlwp_get_instance()->vlwp_test_remote_connection();
					if ( ! empty( $result['success'] ) ) {
						$main_notice = array( 'setting' => self::PAGE_SLUG, 'code' => 'settings_updated', 'message' => __( 'Settings saved.', 'vlwp-backup' ), 'type' => 'updated' );
					} else {
						$msg = isset( $result['message'] ) ? (string) $result['message'] : __( 'External storage connection failed.', 'vlwp-backup' );
						$main_notice = array( 'setting' => self::PAGE_SLUG, 'code' => 'vlwp_backup_remote_error_main', 'message' => $msg, 'type' => 'error' );
					}
				} else {
					$main_notice = array( 'setting' => self::PAGE_SLUG, 'code' => 'settings_updated', 'message' => __( 'Settings saved.', 'vlwp-backup' ), 'type' => 'updated' );
				}

				// Filter out existing settings_updated notices
				$filtered = array();
				if ( is_array( $errors ) ) {
					foreach ( $errors as $e ) {
						if ( isset( $e['code'] ) && 'settings_updated' === $e['code'] ) {
							continue;
						}
						$filtered[] = $e;
					}
				}

				array_unshift( $filtered, $main_notice );

				// Print merged notices
				foreach ( $filtered as $e ) {
					$class = ( isset( $e['type'] ) && 'error' === $e['type'] ) ? 'notice notice-error is-dismissible' : 'notice notice-success is-dismissible';
					echo '<div class="' . esc_attr( $class ) . '"><p>' . esc_html( $e['message'] ) . '</p></div>';
				}

				$printed = true;
			}

			if ( ! $printed ) {
				// default rendering of any existing settings errors/notices
				settings_errors();
			}
      ?>
			   <form method="post" action="options.php">
				   <ul class="vlwp-backup-settings-fields">
				   <?php
				   settings_fields( self::PAGE_SLUG );
				   do_settings_sections( self::PAGE_SLUG );
				   ?>
				   </ul>
				   <?php submit_button(); ?>
			   </form>
			   <script type="text/javascript">
			   (function(){
				   function toggleExternalFields(){
					   var sel = document.querySelector('select[name="vlwp_backup_settings[storage_mode]"]');
					   if(!sel) return;
					   var mode = sel.value;
					   var wrappers = document.querySelectorAll('.vlwp-external-field');
					   Array.prototype.forEach.call(wrappers, function(w){
						   var modes = (w.getAttribute('data-modes')||'').split(',');
						   var row = w.closest('tr');
						   if(row){
							   row.style.display = 'none';
						   }
						   w.style.display = 'none';
						   if(mode !== 'local' && modes.indexOf(mode) !== -1){
							   if(row){
								   row.style.display = '';
							   }
							   w.style.display = '';
						   }
					   });
				   }
				   document.addEventListener('DOMContentLoaded', function(){
					   var sel = document.querySelector('select[name="vlwp_backup_settings[storage_mode]"]');
					   if(!sel) return;
					   sel.addEventListener('change', toggleExternalFields);
					   toggleExternalFields();
				   });
			   })();
			   </script>
		</div>
		<?php
	}

	/**
	 * Register dashboard widget.
	 */
	public function vlwp_register_dashboard_widget() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		wp_add_dashboard_widget(
			'vlwp_backup_status_widget',
			__( 'Backup', 'vlwp-backup' ),
			array( $this, 'vlwp_render_dashboard_widget' )
		);
	}

	/**
	 * Render dashboard widget content.
	 */
	public function vlwp_render_dashboard_widget() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$settings = VLWP_Backup_Core::vlwp_get_instance()->vlwp_get_settings();
		// If no schedule set, set default to enabled, 24h
		if ( ! isset( $settings['schedule_enabled'] ) ) {
			$settings['schedule_enabled'] = 1;
			$settings['schedule_interval_value'] = 1;
			$settings['schedule_interval_unit'] = 'days';
			update_option( vlwp_backup_option_key, $settings );
		}
		$auto_enabled = ! empty( $settings['schedule_enabled'] );
		$auto_color = $auto_enabled ? 'green' : 'red';
		$auto_label = $auto_enabled ? esc_html__( 'Enabled', 'vlwp-backup' ) : esc_html__( 'Disabled', 'vlwp-backup' );
		$auto_title = $auto_enabled ? esc_attr__( 'Click to disable auto-backups', 'vlwp-backup' ) : esc_attr__( 'Click to enable auto-backups', 'vlwp-backup' );

		echo '<p style="margin-bottom:6px;"><strong>' . esc_html__( 'Auto-backups:', 'vlwp-backup' ) . '</strong> <a href="#" id="vlwp-auto-backups-toggle" style="color:' . $auto_color . ';font-weight:600;text-decoration:underline;cursor:pointer;" title="' . $auto_title . '"><span id="vlwp-auto-backups-status">' . $auto_label . '</span></a></p>';

		$index = VLWP_Backup_Storage::vlwp_get_local_backups();
		$total = is_array( $index ) ? count( $index ) : 0;
		$last_mtime = 0;
		foreach ( $index as $name => $meta ) {
			if ( isset( $meta['mtime'] ) && (int) $meta['mtime'] > $last_mtime ) {
				$last_mtime = (int) $meta['mtime'];
			}
		}
		$last_display = $last_mtime ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_mtime ) : __( 'Never', 'vlwp-backup' );

		$status = get_option( 'vlwp_backup_status', array() );
		$ok = isset( $status['ok'] ) ? (bool) $status['ok'] : false;
		$checked_at = isset( $status['checked_at'] ) ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $status['checked_at'] ) : __( 'Never', 'vlwp-backup' );
		$indicator = $ok ? '<span style="color:green;font-weight:600;">' . esc_html__( 'OK', 'vlwp-backup' ) . '</span>' : '<span style="color:red;font-weight:600;">' . esc_html__( 'FAILED', 'vlwp-backup' ) . '</span>';

		// External storage status
		$external_status = get_option( 'vlwp_backup_external_status', array() );
		$external_ok = isset($external_status['ok']) ? (bool)$external_status['ok'] : false;
		$external_checked_at = isset($external_status['checked_at']) ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $external_status['checked_at']) : __( 'Never', 'vlwp-backup' );
		$external_indicator = $external_ok ? '<span style="color:green;font-weight:600;">' . esc_html__( 'OK', 'vlwp-backup' ) . '</span>' : '<span style="color:red;font-weight:600;">' . esc_html__( 'FAILED', 'vlwp-backup' ) . '</span>';

		$all_link = esc_url( admin_url( 'admin.php?page=vlwp-backup' ) );
		$settings_link = esc_url( admin_url( 'admin.php?page=vlwp-backup_settings' ) );

		echo '<p><strong>' . esc_html__( 'Total backups:', 'vlwp-backup' ) . '</strong> ' . esc_html( $total ) . ' (<a href="' . $all_link . '">' . esc_html__( 'View all', 'vlwp-backup' ) . '</a>)</p>';
		echo '<p><strong>' . esc_html__( 'Last backup:', 'vlwp-backup' ) . '</strong> ' . esc_html( $last_display ) . '</p>';
		echo '<p><strong>' . esc_html__( 'Last check:', 'vlwp-backup' ) . '</strong> ' . $indicator . ' — ' . esc_html( $checked_at ) . '</p>';

		echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin-top:8px;">';
		echo '<input type="hidden" name="action" value="vlwp_backup_run_status_check" />';
		wp_nonce_field( 'vlwp_backup_run_status_check' );
		submit_button( __( 'Run backup check', 'vlwp-backup' ), 'secondary', 'submit', false );
		echo '</form>';

		echo '<p><strong>' . esc_html__( 'External storage status:', 'vlwp-backup' ) . '</strong> ' . $external_indicator . ' — ' . esc_html( $external_checked_at ) . '</p>';

		echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin-top:8px;">';
		echo '<input type="hidden" name="action" value="vlwp_backup_run_external_check" />';
		wp_nonce_field( 'vlwp_backup_run_external_check' );
		submit_button( __( 'Run external storage check', 'vlwp-backup' ), 'secondary', 'submit', false );
		echo '</form>';

		// Inline JS for toggle
		?>
		<script type="text/javascript">
		(function(){
			var toggle = document.getElementById('vlwp-auto-backups-toggle');
			if(!toggle) return;
			toggle.addEventListener('click', function(e){
				e.preventDefault();
				var status = document.getElementById('vlwp-auto-backups-status');
				var orig = status.textContent;
				status.textContent = '...';
				fetch(ajaxurl, {
					method: 'POST',
					credentials: 'same-origin',
					headers: {'Content-Type': 'application/x-www-form-urlencoded'},
					body: 'action=vlwp_toggle_auto_backups&nonce=' + encodeURIComponent('<?php echo wp_create_nonce('vlwp_toggle_auto_backups'); ?>')
				})
				.then(function(r){return r.json();})
				.then(function(data){
					if(data.success){
						if(data.data.enabled){
							status.textContent = 'Enabled';
							toggle.style.color = 'green';
							toggle.title = 'Click to disable auto-backups';
						} else{
							status.textContent = 'Disabled';
							toggle.style.color = 'red';
							toggle.title = 'Click to enable auto-backups';
						}
					} else{
						status.textContent = orig;
						alert(data.data && data.data.message ? data.data.message : 'Error');
					}
				})
				.catch(function(){
					status.textContent = orig;
					alert('Error');
				});
			});
		})();
		</script>
		<?php
	}

  /**
   * Handle dashboard "Run backup check" action.
   */
  public function vlwp_handle_run_status_check() {
      if ( ! current_user_can( 'manage_options' ) ) {
          wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
      }

      check_admin_referer( 'vlwp_backup_run_status_check' );

      // Trigger the background status check action immediately.
      do_action( 'vlwp_backup_check_status_cron' );

      $redirect = wp_get_referer() ? wp_get_referer() : admin_url( 'index.php' );
      $redirect = add_query_arg( 'vlwp_backup_notice', 'success', $redirect );
      wp_safe_redirect( $redirect );
      exit;

  }

  /**
   * Handle dashboard "Run external storage check" action.
   */
  public function vlwp_handle_run_external_check() {
      if ( ! current_user_can( 'manage_options' ) ) {
          wp_die( esc_html__( 'Not allowed.', 'vlwp-backup' ) );
      }
      check_admin_referer( 'vlwp_backup_run_external_check' );
      // Trigger the external storage check action immediately.
      $this->vlwp_check_external_status();
      $redirect = wp_get_referer() ? wp_get_referer() : admin_url( 'index.php' );
      $redirect = add_query_arg( 'vlwp_backup_notice', 'success', $redirect );
      wp_safe_redirect( $redirect );
      exit;
  }

}
